import {MutableRefObject, useEffect, useMemo, useRef, useState} from 'react';

import {isNotUndefined} from 'types/utilities';
import {
    ESearchFormFieldName,
    ISearchFormValues,
    TSearchFormErrors,
} from 'components/SearchForm/types';
import {EPubSubEvent} from 'types/EPubSubEvent';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import useImmutableCallback from 'utilities/hooks/useImmutableCallback';
import {usePubSubSubscriber} from 'utilities/hooks/usePubSubSubscriber';

type TNodesObject = PartialRecord<ESearchFormFieldName, HTMLElement>;
type TNodesRefsObject = PartialRecord<
    ESearchFormFieldName,
    MutableRefObject<HTMLElement>
>;

export interface ISearchFormFieldsRefsAndErrors {
    focusFieldByName(fieldName: ESearchFormFieldName): void;
    getFieldRefByName(fieldName: ESearchFormFieldName): HTMLElement | undefined;
    setFieldRefByName(
        fieldName: ESearchFormFieldName,
    ): (rootNode: HTMLElement) => void;
    setErrorFieldRefByName(
        fieldName: ESearchFormFieldName,
    ): (rootNode: HTMLElement) => void;
    focusNextField(): void;
    focusNextFieldByName(nextFieldName: ESearchFormFieldName): void;
    handleBlurField(fieldName: ESearchFormFieldName): void;
    handleFocusField(fieldName: ESearchFormFieldName): void;
    handleTrySubmit(): void;

    errorFieldRef: MutableRefObject<HTMLElement> | undefined;
    errorTooltipText: string | undefined;
}

export default function useFieldsRefsAndErrors({
    fieldsNames,
    autoFocusFieldName,
    formErrors,
    formValues,
    canShowErrors,
}: {
    fieldsNames: ESearchFormFieldName[];
    autoFocusFieldName?: ESearchFormFieldName;
    formErrors: TSearchFormErrors;
    formValues: ISearchFormValues;
    canShowErrors: boolean;
}): ISearchFormFieldsRefsAndErrors {
    const deviceType = useDeviceType();

    const [currentFocusField, setCurrentFocusField] =
        useState<ESearchFormFieldName | null>(null);

    const fieldNodesRef = useRef<TNodesObject>({});
    const errorFieldRefs = useRef<TNodesRefsObject>({});

    const getFieldNodes = useImmutableCallback(() => {
        return fieldNodesRef.current;
    });

    const getFieldRefByName = useImmutableCallback(
        (fieldName: ESearchFormFieldName) => {
            const fieldNodes = getFieldNodes();

            return fieldNodes[fieldName];
        },
    );

    const getFieldNameByNode = useImmutableCallback(
        (fieldNode: EventTarget | null) => {
            const allFieldsNodes = getFieldNodes();
            const allFieldsNames = Object.keys(
                allFieldsNodes,
            ) as ESearchFormFieldName[];
            let currentFieldName = null;

            allFieldsNames.some(fieldName => {
                const isCurrentField = fieldNode === allFieldsNodes[fieldName];

                if (isCurrentField) {
                    currentFieldName = fieldName;
                }

                return isCurrentField;
            });

            return currentFieldName;
        },
    );

    const getErrorFieldRefs = useImmutableCallback(() => {
        return errorFieldRefs.current;
    });

    const getErrorFieldRefByName = useImmutableCallback(
        (fieldName: ESearchFormFieldName) => {
            const errorFieldsNodes = getErrorFieldRefs();

            return errorFieldsNodes[fieldName];
        },
    );

    const focusFieldByName = useImmutableCallback(
        (fieldName: ESearchFormFieldName) => {
            const fieldNode = getFieldRefByName(fieldName);

            if (fieldNode && typeof fieldNode.focus === 'function') {
                requestAnimationFrame(() => fieldNode.focus());
            }
        },
    );

    /**
     * Проверяет значение query-параметра "focus" и выставляет фокус соответствующему
     * полю поисковой формы.
     **/
    const checkFocusQueryParam = useImmutableCallback(() => {
        if (deviceType.isMobile) {
            return;
        }

        if (autoFocusFieldName) {
            focusFieldByName(autoFocusFieldName);
        }
    });

    const focusFirstErrorField = useImmutableCallback(() => {
        fieldsNames.some(fieldName => {
            const isErrorField = formErrors[fieldName];

            if (isErrorField) {
                focusFieldByName(fieldName);
            }

            return isErrorField;
        });
    });

    const isEmptyField = useImmutableCallback(
        (fieldName: ESearchFormFieldName): boolean => {
            const fieldValue = formValues[fieldName];

            if (fieldValue === null) {
                return true;
            }

            if (typeof fieldValue === 'object') {
                return fieldValue.inputValue === '';
            }

            return !fieldValue;
        },
    );

    const focusNextFieldByName = useImmutableCallback(
        (nextFieldName: ESearchFormFieldName): void => {
            const nextFieldIndex = fieldsNames.indexOf(nextFieldName);

            if (nextFieldIndex !== -1) {
                for (let i = nextFieldIndex; i < fieldsNames.length; i++) {
                    const nextField = fieldsNames[i];
                    const canFocusField =
                        formErrors[nextField] || isEmptyField(nextField);

                    if (canFocusField) {
                        focusFieldByName(nextField);

                        break;
                    }
                }
            }
        },
    );

    const focusNextField = useImmutableCallback((): void => {
        if (!currentFocusField) {
            return;
        }

        const currentFocusFieldIndex = fieldsNames.indexOf(currentFocusField);

        if (currentFocusFieldIndex !== -1) {
            const nextFieldName = fieldsNames[currentFocusFieldIndex + 1];

            if (nextFieldName) {
                focusFieldByName(nextFieldName);
            }
        }
    });

    const handleFocusListener = useImmutableCallback((e: Event) => {
        setCurrentFocusField(getFieldNameByNode(e.target));
    });

    const handleBlurListener = useImmutableCallback(() => {
        setCurrentFocusField(null);
    });

    const handleFocusField = useImmutableCallback(
        (fieldName: ESearchFormFieldName): void => {
            setCurrentFocusField(fieldName);
        },
    );

    const handleTrySubmit = useImmutableCallback(() => {
        focusFirstErrorField();
    });

    const handleBlurField = useImmutableCallback(
        (fieldName: ESearchFormFieldName) => {
            if (currentFocusField === fieldName) {
                setCurrentFocusField(null);
            }
        },
    );

    const removeFieldListeners = useImmutableCallback(
        (fieldNode: HTMLElement) => {
            if (fieldNode) {
                fieldNode.removeEventListener('focus', handleFocusListener);
                fieldNode.removeEventListener('blur', handleBlurListener);
            }
        },
    );

    const removeAllListeners = useImmutableCallback(() => {
        const allFieldsNodes = getFieldNodes();

        Object.values(allFieldsNodes)
            .filter(isNotUndefined)
            .forEach(removeFieldListeners);
    });

    const addFieldListeners = useImmutableCallback((fieldNode: HTMLElement) => {
        if (fieldNode) {
            fieldNode.addEventListener('focus', handleFocusListener);
            fieldNode.addEventListener('blur', handleBlurListener);
        }
    });

    const setErrorFieldRefByName = useImmutableCallback(
        (fieldName: ESearchFormFieldName) =>
            (rootNode: HTMLElement): void => {
                const errorFieldsNodes = getErrorFieldRefs();

                if (rootNode) {
                    errorFieldsNodes[fieldName] = errorFieldsNodes[
                        fieldName
                    ] || {
                        current: rootNode,
                    };

                    const fieldNodeRef = errorFieldsNodes[fieldName];

                    if (fieldNodeRef) {
                        fieldNodeRef.current = rootNode;
                    }
                }
            },
    );

    const setFieldRefByName = useImmutableCallback(
        (fieldName: ESearchFormFieldName) => {
            return (rootNode: HTMLElement): void => {
                const fieldNodes = getFieldNodes();

                if (rootNode) {
                    fieldNodes[fieldName] = rootNode;

                    removeFieldListeners(rootNode);
                    addFieldListeners(rootNode);
                }
            };
        },
    );

    const {errorFieldRef, errorTooltipText} = useMemo(() => {
        if (!currentFocusField) {
            return {};
        }

        const fieldErrors = formErrors[currentFocusField];

        if (
            deviceType.isDesktop &&
            canShowErrors &&
            currentFocusField &&
            fieldErrors
        ) {
            return {
                errorFieldRef: getErrorFieldRefByName(currentFocusField),
                errorTooltipText: fieldErrors[0],
            };
        }

        return {};
    }, [
        canShowErrors,
        currentFocusField,
        deviceType.isDesktop,
        formErrors,
        getErrorFieldRefByName,
    ]);

    usePubSubSubscriber(EPubSubEvent.FOCUS_SEARCH_FORM_FIELD, fieldName => {
        focusFieldByName(fieldName);
    });

    useEffect(() => {
        requestAnimationFrame(checkFocusQueryParam);
    }, [checkFocusQueryParam]);

    useEffect(() => {
        return (): void => {
            removeAllListeners();
        };
    }, [removeAllListeners]);

    return {
        handleTrySubmit,
        handleFocusField,
        handleBlurField,
        focusFieldByName,
        getFieldRefByName,
        setFieldRefByName,
        setErrorFieldRefByName,
        focusNextField,
        focusNextFieldByName,

        errorFieldRef,
        errorTooltipText,
    };
}
