import {Component, ReactNode} from 'react';

import * as i18nBlock from 'i18n/components-search-form';

/* Components */
import Button from 'components/Button/Button';
import ArrowLongRightIcon from 'icons/16/ArrowLongRight';

/* Init Styles */
import cx from './SearchFormFieldModal.scss';

/* Constants */
const DEFAULT_BUTTONS: ISearchFormFieldModalButton[] = [
    {type: 'complete', text: i18nBlock.searchModalComplete()},
    {type: 'next', text: i18nBlock.searchModalNext()},
];

/* Components Types */
interface ISearchFormFieldModalButton {
    type: string;
    text: string;
}

interface ISearchFormFieldModalProps {
    isLastControl?: boolean;
    buttonsAreVisible: boolean;
    cancelButtonIsVisible: boolean;
    fieldType: string;
    triggerNode: ReactNode;
    componentNode: ReactNode;
    error?: string;
    buttons: ISearchFormFieldModalButton[];
    onHideModal: (fieldType: string) => void;
    onCancelClick: (fieldType: string) => void;
    onButtonClick?: (params: {buttonType: string; fieldType: string}) => void;
}

class SearchFormFieldModal extends Component<ISearchFormFieldModalProps> {
    static readonly defaultProps: Partial<ISearchFormFieldModalProps> = {
        buttons: DEFAULT_BUTTONS,
        buttonsAreVisible: true,
    };

    /* Handlers */
    private handleFooterButtonClick = (buttonType: string) => {
        const {fieldType, onHideModal, onButtonClick} = this.props;

        if (onHideModal) {
            onHideModal(fieldType);
        }

        if (onButtonClick) {
            onButtonClick({buttonType, fieldType});
        }
    };

    private handleCancelButtonClick = () => {
        const {fieldType, onHideModal, onCancelClick} = this.props;

        if (onHideModal) {
            onHideModal(fieldType);
        }

        if (onCancelClick) {
            onCancelClick(fieldType);
        }
    };

    /* Render */
    private renderHeader() {
        const {triggerNode, cancelButtonIsVisible} = this.props;

        return (
            <div className={cx('trigger')}>
                {cancelButtonIsVisible && (
                    <button
                        className={cx('cancelButton')}
                        type="button"
                        onClick={this.handleCancelButtonClick}
                    >
                        <ArrowLongRightIcon
                            className={cx('backIcon')}
                            width={20}
                            height={64}
                        />
                    </button>
                )}
                {triggerNode}
            </div>
        );
    }

    private renderSearchFormComponent() {
        const {componentNode, buttonsAreVisible} = this.props;

        return (
            <div
                className={cx('component', {
                    component_withButtons: buttonsAreVisible,
                })}
            >
                {componentNode}
            </div>
        );
    }

    private renderButton = (
        button: ISearchFormFieldModalButton,
        index: number,
    ) => {
        const {type, text} = button;

        return (
            <Button
                key={index}
                className={cx('button')}
                onClick={this.handleFooterButtonClick.bind(this, type)}
                size="l"
                width="max"
                theme="primary"
            >
                {text}
            </Button>
        );
    };

    private renderButtons() {
        const {buttonsAreVisible, buttons, isLastControl} = this.props;

        if (!buttonsAreVisible) {
            return null;
        }

        return (
            <div className={cx('buttons')}>
                {buttons
                    .filter(({type}) => !(isLastControl && type === 'next'))
                    .map(this.renderButton)}
            </div>
        );
    }

    private renderErrorLabel() {
        const {error} = this.props;

        if (error) {
            return <div className={cx('errorTooltip')}>{error}</div>;
        }
    }

    private renderFooter() {
        return (
            <div className={cx('footer')}>
                {this.renderErrorLabel()}
                {this.renderButtons()}
            </div>
        );
    }

    render() {
        return (
            <div className={cx('searchModal')}>
                <div className={cx('content')}>
                    {this.renderHeader()}
                    {this.renderSearchFormComponent()}
                    {this.renderFooter()}
                </div>
            </div>
        );
    }
}

export default SearchFormFieldModal;
