import React from 'react';
import _noop from 'lodash/noop';

import {IWithClassName} from 'types/IWithClassName';

import Select from 'components/SelectOld/SelectOld';
import CloseCircleIcon from 'icons/16/CloseCircle';

import cx from './TravellersSelect.scss';

const SelectButton = Select.SelectButton;

interface ITravellersSelectItem {
    value: number;
    text: string;
}

interface ITravellersSelectProps extends IWithClassName {
    type: string;
    values: number[];
    items: ITravellersSelectItem[];
    isNative?: boolean;
    nativeSelectProps: React.DetailedHTMLProps<
        React.SelectHTMLAttributes<HTMLSelectElement>,
        HTMLSelectElement
    >;
    buttonLabel: string;
    maxValuesCount: number;
    renderSelectedItem(value: number): string;
    onChange(v: {type: string; values: number[]}): void;
}

class TravellersSelect extends React.Component<ITravellersSelectProps> {
    static readonly defaultProps = {
        type: '',
        items: [],
        values: [],
        isNative: true,
        onChange: _noop,
        nativeSelectProps: {},
        renderSelectedItem: _noop,
    };

    private renderButtonComponent = ({
        meta = {},
        input = {},
    }: {
        meta: {
            visible?: boolean;
            hidePopup?: () => void;
            showPopup?: () => void;
        };
        input: {ref?: React.RefObject<HTMLDivElement>};
    }): React.ReactElement => {
        const {buttonLabel} = this.props;
        const {visible, hidePopup, showPopup} = meta;

        const {ref} = input;

        return (
            <SelectButton
                className={cx('selectButton')}
                buttonRef={ref}
                onClick={visible ? hidePopup : showPopup}
            >
                {buttonLabel}
            </SelectButton>
        );
    };

    private handleSelectChange = ({
        type,
        value,
    }: {
        type: string;
        value: number | string;
    }): void => {
        const {onChange, values} = this.props;
        const preparedValues = [...values, Number(value)];

        onChange({type, values: preparedValues});
    };

    private handleResetValueClick(
        value: number,
        e: React.MouseEvent<HTMLDivElement>,
    ): void {
        const {onChange, type, values} = this.props;
        const preparedValues = [...values];
        const valueIndex = preparedValues.indexOf(value);

        preparedValues.splice(valueIndex, 1);

        onChange({type, values: preparedValues});

        e.stopPropagation();
    }

    private renderSelectedValue = (
        value: number,
        index: number,
    ): React.ReactNode => {
        const {renderSelectedItem} = this.props;

        return (
            <div className={cx('selectedValue')} key={`${value}-${index}`}>
                <div className={cx('selectedValueContent')}>
                    {renderSelectedItem(value)}
                </div>
                <div
                    className={cx('selectedValueReset')}
                    onClick={this.handleResetValueClick.bind(this, value)}
                >
                    <CloseCircleIcon width={18} height={18} />
                </div>
            </div>
        );
    };

    private renderSelectedValues(): React.ReactNode {
        const {values} = this.props;

        return values.map(this.renderSelectedValue);
    }

    private renderSelect(): React.ReactNode {
        const {values, maxValuesCount, items, type, nativeSelectProps} =
            this.props;

        const selectedValuesCount = values.length;
        const canRenderSelect = selectedValuesCount < maxValuesCount;

        return (
            canRenderSelect && (
                <Select
                    type={type}
                    items={items}
                    onChange={this.handleSelectChange}
                    switcherComponent={this.renderButtonComponent}
                    nativeSelectProps={nativeSelectProps}
                />
            )
        );
    }

    render(): React.ReactNode {
        const {className} = this.props;

        return (
            <div className={cx('travellersSelect', className)}>
                {this.renderSelectedValues()}
                {this.renderSelect()}
            </div>
        );
    }
}

export default TravellersSelect;
