import {FC, StrictMode, useEffect} from 'react';
import {useInView} from 'react-intersection-observer';

import {ZINDEX_GROUP_LEVEL} from 'constants/layers';

import {IDevice} from 'types/IDevice';
import {IMetrika} from 'types/metrika/IMetrika';
import {EMetrikaGoal} from 'types/metrika/EMetrikaGoal';
import {IErrorBooster} from 'types/errorBooster/IErrorBooster';
import {IAffiliateParams} from 'types/affiliate/IAffiliateParams';

import ErrorBoundary from 'components/ErrorBoundary/ErrorBoundary';
import BottomSheetManager from 'components/BottomSheet/components/BottomSheetManager/BottomSheetManager';

import LayersContext from 'contexts/LayersContext';
import MetrikaContext from 'contexts/MetrikaContext';
import AffiliateContext from 'contexts/AffiliateContext';
import DeviceTypeContext from 'contexts/DeviceTypeContext';

import cx from './Widget.scss';

interface IWidgetProps {
    affiliate: IAffiliateParams;
    deviceType: IDevice;
    metrika: IMetrika;
    errorBooster: IErrorBooster;
}

const Widget: FC<IWidgetProps> = params => {
    const {children, affiliate, metrika, errorBooster, deviceType} = params;
    const {ref, inView} = useInView({triggerOnce: true, threshold: 0.5});

    useEffect(() => {
        metrika.reachGoal(EMetrikaGoal.WIDGET_MOUNTED);
    }, [metrika]);

    useEffect(() => {
        if (inView) {
            metrika.reachGoal(EMetrikaGoal.WIDGET_SHOWN);
        }
    }, [inView, metrika]);

    return (
        <StrictMode>
            <ErrorBoundary errorBooster={errorBooster}>
                <AffiliateContext.Provider value={affiliate}>
                    <MetrikaContext.Provider value={metrika}>
                        <DeviceTypeContext.Provider value={deviceType}>
                            <LayersContext.Provider value={ZINDEX_GROUP_LEVEL}>
                                <BottomSheetManager>
                                    <div className={cx('root')} ref={ref}>
                                        {children}
                                    </div>
                                </BottomSheetManager>
                            </LayersContext.Provider>
                        </DeviceTypeContext.Provider>
                    </MetrikaContext.Provider>
                </AffiliateContext.Provider>
            </ErrorBoundary>
        </StrictMode>
    );
};

export default Widget;
