import React, {useCallback, useEffect, useMemo, useRef, useState} from 'react';

import {HOTELS_SEARCH_FORM_NAME} from 'constants/searchForm';

import {IWithClassName} from 'types/IWithClassName';
import {
    ESearchFormFieldName,
    ESearchFormSize,
    ESearchFormTriggerViewType,
} from 'components/SearchForm/types';
import {THotelSlug} from 'types/hotels/hotel/IHotel';
import {ECalendarType} from 'components/Calendar/types';
import {IHotelsSuggest} from 'types/hotels/common/ISearchSuggest';
import {ISelectItemPreparedOptions} from 'components/SearchForm/types/ISelectItemOptions';
import {ISearchFormRenderTravelersBlockProps} from 'components/SearchForm/types/ISearchFormRenderTravelersBlockProps';
import {IHotelsSearchFormValues} from 'projects/hotels/components/SearchForm/types';

import validateForm from 'projects/hotels/components/SearchForm/utilities/validateForm';
import renderSuggestItemTitle from 'projects/hotels/components/SearchForm/utilities/renderSuggestItemTitle';
import renderSuggestItemDescription from 'projects/hotels/components/SearchForm/utilities/renderSuggestItemDescription';
import getSuggestTitleAndDescription from 'projects/hotels/components/SearchForm/utilities/getSuggestTitleAndDescription';

import * as i18nBlock from 'i18n/hotels-search-form';

import useImmutableCallback from 'hooks/useImmutableCallback';
import {useDeviceType} from 'hooks/useDeviceType';

import CommonSearchForm, {
    IBaseSearchFormProps,
    IOnDateClickParams,
} from 'components/SearchForm/SearchForm';
import useRequestSuggestsWithThrottle from 'components/SearchForm/hooks/useRequestSuggestsWithThrottle';
import useHandleSubmit from 'projects/hotels/components/SearchForm/hooks/useHandleSubmit';
import HotelsTravellers from 'projects/hotels/components/SearchForm/components/HotelsTravellers/HotelsTravellers';
import {
    ISuggestValue,
    TSelectedValue,
} from 'components/SearchSuggest/SearchSuggest';
import {IDatePickerFooterProps} from 'components/DatePicker/components/DatePickerFooter/DatePickerFooter';

const FIELD_NAMES = [
    ESearchFormFieldName.TO,
    ESearchFormFieldName.START_DATE,
    ESearchFormFieldName.END_DATE,
];

export interface ISearchFormProps extends IBaseSearchFormProps, IWithClassName {
    activeHotelSlug?: THotelSlug;
    queryParams?: Record<string, string>;
    formRef?: React.RefObject<HTMLFormElement>;
    onSubmit?(params: {searchLink: string}): void;

    searchForm: IHotelsSearchFormValues;
    searchSuggests: IHotelsSuggest[];

    setToPoint(fieldValue: ISuggestValue<IHotelsSuggest>): void;
    setStartDate(date: string | null): void;
    setEndDate(date: string | null): void;
    setAdults(count: number): void;
    setChildrenAges(ages: number[]): void;

    requestSuggests(fieldInputValue: string): void;
    onSelectSuggestItem?(
        item: TSelectedValue<IHotelsSuggest>,
        options?: ISelectItemPreparedOptions,
    ): void;
}

const canFocusNextField = (): boolean => false;

const SearchForm: React.FC<ISearchFormProps> = props => {
    const {
        isExpanded,
        size = ESearchFormSize.XL,
        triggerViewType = ESearchFormTriggerViewType.UNION,
        activeHotelSlug,
        canSubmitOnChange = false,
        canToggleDropdowns = false,
        fieldsRef,
        onShowFieldPopup,
        onHideFieldPopup,
        onSubmit,
        queryParams,
        formRef,

        searchForm,
        searchSuggests,

        setToPoint,
        setStartDate,
        setEndDate,
        setAdults,
        setChildrenAges,

        requestSuggests,
        onSelectSuggestItem,

        ...restProps
    } = props;

    const [travellersScope, setTravellersScope] = useState<HTMLDivElement>();
    const travellersRef = useRef<HTMLDivElement>(null);

    const {
        to: toField,
        startDate: startDateField,
        endDate: endDateField,
        adults: adultsField,
        childrenAges: childrenAgesField,
    } = searchForm;

    const deviceType = useDeviceType();

    const requestSuggestsWithThrottle =
        useRequestSuggestsWithThrottle(requestSuggests);

    const getTravelersBlurHandler = useImmutableCallback(
        (submit: () => void) => {
            return (): void => {
                if (canSubmitOnChange) {
                    requestAnimationFrame(() => {
                        submit();
                    });
                }
            };
        },
    );

    const handleSubmit = useHandleSubmit({
        toField,
        startDateField,
        endDateField,
        adultsField,
        childrenAgesField,
        activeHotelSlug,
        queryParams,
        onSubmit,
    });

    const handleDateClick = useImmutableCallback(
        (args?: IOnDateClickParams) => {
            if (!args) {
                return;
            }

            const {startDate, endDate, calendarType} = args;

            if (startDate && endDate && Number(startDate) === Number(endDate)) {
                if (calendarType === ECalendarType.START_DATE) {
                    setEndDate(null);
                } else {
                    setStartDate(null);
                }
            }
        },
    );

    useEffect(() => {
        if (travellersRef.current) {
            setTravellersScope(travellersRef.current);
        }
    }, [travellersRef.current]);

    useEffect(() => {
        requestSuggestsWithThrottle(toField.inputValue);
    }, [requestSuggestsWithThrottle, toField.inputValue]);

    const renderTravellersBlock = useCallback(
        ({baseCx, submit}: ISearchFormRenderTravelersBlockProps) => {
            return (
                <div className={baseCx('travellers')} ref={travellersRef}>
                    <HotelsTravellers
                        scope={travellersScope}
                        triggerClassName={baseCx(
                            'travellersTrigger',
                            'lastControl',
                        )}
                        triggerFocusClassName={baseCx(
                            'travellersTrigger_focus',
                        )}
                        size={size}
                        triggerViewType={triggerViewType}
                        tabIndex={isExpanded ? 0 : -1}
                        adultsCount={adultsField}
                        setAdults={setAdults}
                        childrenAges={childrenAgesField}
                        setChildrenAges={setChildrenAges}
                        deviceType={deviceType}
                        canToggle
                        onShowPopup={onShowFieldPopup}
                        onHidePopup={onHideFieldPopup}
                        onBlur={getTravelersBlurHandler(submit)}
                    />
                </div>
            );
        },
        [
            travellersRef,
            travellersScope,
            adultsField,
            childrenAgesField,
            deviceType,
            getTravelersBlurHandler,
            isExpanded,
            onHideFieldPopup,
            onShowFieldPopup,
            setAdults,
            setChildrenAges,
            size,
            triggerViewType,
        ],
    );

    const footerBlockParams = useMemo((): IDatePickerFooterProps => {
        let messageText;

        if (!startDateField) {
            messageText = i18nBlock.selectCheckInDate();
        } else if (!endDateField) {
            messageText = i18nBlock.selectCheckOutDate();
        }

        const buttonText = i18nBlock.selectButton();
        const visible = deviceType.isMobile;
        const showButton = true;
        const buttonDisabled = !startDateField || !endDateField;

        return {visible, messageText, showButton, buttonText, buttonDisabled};
    }, [startDateField, endDateField, deviceType.isMobile]);

    return (
        <CommonSearchForm
            size={size}
            triggerViewType={triggerViewType}
            searchFormName={HOTELS_SEARCH_FORM_NAME}
            fieldsNames={FIELD_NAMES}
            toField={toField}
            startDateField={startDateField}
            endDateField={endDateField}
            hasEndDate
            toSearchSuggests={searchSuggests}
            validateForm={validateForm}
            uniqueSuggestValueName="id"
            getSuggestTitleAndDescription={getSuggestTitleAndDescription}
            renderSuggestItemTitle={renderSuggestItemTitle}
            renderSuggestItemDescription={renderSuggestItemDescription}
            toPointPlaceholder={i18nBlock.hotelsSearchFormDotToFieldPlaceholder()}
            startDatePlaceholder={i18nBlock.hotelsSearchFormDotStartDatePlaceholder()}
            endDatePlaceholder={i18nBlock.hotelsSearchFormDotEndDatePlaceholder()}
            setToPoint={setToPoint}
            setStartDate={setStartDate}
            setEndDate={setEndDate}
            onSubmit={handleSubmit}
            isRoundTrip
            isExpanded={isExpanded}
            renderTravellersBlock={renderTravellersBlock}
            canSubmitOnChange={canSubmitOnChange}
            canFocusNextField={
                deviceType.isMobile ? canFocusNextField : undefined
            }
            fieldsRef={fieldsRef}
            onSelectSuggestItem={onSelectSuggestItem}
            canToggleDropdowns={canToggleDropdowns}
            onShowFieldPopup={onShowFieldPopup}
            onHideFieldPopup={onHideFieldPopup}
            onDateClick={handleDateClick}
            formRef={formRef}
            datePickerFooterBlockParams={footerBlockParams}
            {...restProps}
        />
    );
};

export default React.memo(SearchForm);
