import React, {useCallback, useEffect, useMemo, useReducer} from 'react';
import {widgetsApi} from 'api/widgetsApi';

import {
    HOTELS_SUGGEST_DEFAULT_DOMAIN,
    HOTELS_SUGGEST_DEFAULT_LANGUAGE,
    SEARCH_SUGGEST_LIMIT,
} from 'constants/hotels';

import {IHotelsSuggest} from 'types/hotels/common/ISearchSuggest';
import {TDate} from 'types/common/IDate';
import {EMetrikaGoal} from 'types/metrika/EMetrikaGoal';

import isNearbySuggest from 'components/SearchSuggest/utilities/isNearbySuggest';

import {searchSubmitButton} from 'i18n/components-search-form';

import {useAffiliateParams} from 'hooks/useAffiliateParams';
import useImmutableCallback from 'hooks/useImmutableCallback';
import {useMetrika} from 'hooks/useMetrika';

import {
    ESuggestSource,
    ISuggestValue,
} from 'components/SearchSuggest/SearchSuggest';
import SearchForm from 'projects/hotels/components/SearchForm/SearchForm';

import {
    setHotelsSearchFormAdultsFieldAction,
    setHotelsSearchFormChildrenAgesFieldAction,
    setHotelsSearchFormEndDateFieldAction,
    setHotelsSearchFormStartDateFieldAction,
    setHotelsSearchFormToFieldAction,
} from 'projects/hotels/containers/SearchForm/reducers/searchForm/actions';
import searchFormReducer, {
    SEARCH_FORM_INITIAL_STATE,
} from 'projects/hotels/containers/SearchForm/reducers/searchForm/reducer';
import suggestsReducer, {
    SUGGESTS_INITIAL_STATE,
} from 'projects/hotels/containers/SearchForm/reducers/suggests/reducer';
import {fetchSearchSuggestActions} from 'projects/hotels/containers/SearchForm/reducers/suggests/actions';

export interface IHotelsSearchFormContainer {
    initialPointTo?: {
        hotelSlug?: string;
        regionSlug?: string;
    };
    initialStartDate?: TDate;
    initialEndDate?: TDate;
}

const HotelsSearchFormContainer: React.FC<
    IHotelsSearchFormContainer
> = props => {
    const {initialPointTo, initialStartDate, initialEndDate} = props;
    const affiliateParams = useAffiliateParams();
    const metrika = useMetrika();

    const queryParams = useMemo(
        () => ({
            ...affiliateParams,
        }),
        [affiliateParams],
    );

    const [searchForm, dispatchSearchForm] = useReducer(
        searchFormReducer,
        SEARCH_FORM_INITIAL_STATE,
    );
    const [suggests, dispatchSuggests] = useReducer(
        suggestsReducer,
        SUGGESTS_INITIAL_STATE,
    );
    const {sessionId, requestIndex} = suggests;

    const setToPoint = useCallback(
        (fieldValue: ISuggestValue<IHotelsSuggest>) =>
            dispatchSearchForm(setHotelsSearchFormToFieldAction(fieldValue)),
        [],
    );
    const setStartDate = useCallback(
        (date: string | null) =>
            dispatchSearchForm(setHotelsSearchFormStartDateFieldAction(date)),
        [],
    );
    const setEndDate = useCallback(
        (date: string | null) =>
            dispatchSearchForm(setHotelsSearchFormEndDateFieldAction(date)),
        [],
    );
    const setAdults = useCallback(
        (count: number) =>
            dispatchSearchForm(setHotelsSearchFormAdultsFieldAction(count)),
        [],
    );
    const setChildrenAges = useCallback(
        (ages: number[]) =>
            dispatchSearchForm(
                setHotelsSearchFormChildrenAgesFieldAction(ages),
            ),
        [],
    );

    const handleFormInteraction = useCallback(() => {
        metrika.reachGoal(EMetrikaGoal.SEARCH_FORM_FIELD_CLICK);
    }, [metrika]);

    const handleSubmit = useCallback(() => {
        metrika.reachGoal(EMetrikaGoal.SEARCH_FORM_SUBMIT);
    }, [metrika]);

    const requestSuggests = useImmutableCallback(
        async (options: {
            query?: string;
            regionSlug?: string;
            hotelSlug?: string;
        }): Promise<IHotelsSuggest[]> => {
            dispatchSuggests(fetchSearchSuggestActions.request());

            try {
                const {data} = await widgetsApi.searchSuggest({
                    query: options.query,
                    hotelSlug: options.hotelSlug,
                    regionSlug: options.regionSlug,
                    sessionId,
                    requestIndex,
                    limit: SEARCH_SUGGEST_LIMIT,
                    domain: HOTELS_SUGGEST_DEFAULT_DOMAIN,
                    language: HOTELS_SUGGEST_DEFAULT_LANGUAGE,
                });

                const items = data.items.filter(item => !isNearbySuggest(item));

                dispatchSuggests(
                    fetchSearchSuggestActions.success({
                        items,
                        timeMarker: Date.now(),
                    }),
                );

                return items;
            } catch (err) {
                dispatchSuggests(fetchSearchSuggestActions.failure());

                return [];
            }
        },
    );

    const manualRequestSuggests = useImmutableCallback(
        (fieldInputValue: string) => requestSuggests({query: fieldInputValue}),
    );

    useEffect(() => {
        if (initialStartDate) {
            setStartDate(initialStartDate);
        }

        if (initialEndDate) {
            setEndDate(initialEndDate);
        }

        if (initialPointTo?.hotelSlug || initialPointTo?.regionSlug) {
            (async () => {
                const items = await requestSuggests({
                    hotelSlug: initialPointTo?.hotelSlug,
                    regionSlug: initialPointTo?.regionSlug,
                });

                if (items.length) {
                    setToPoint({
                        selectedValue: items[0],
                        inputValue: items[0].name || '',
                        source: ESuggestSource.SUGGESTS,
                    });
                }
            })();
        }
    }, [
        initialStartDate,
        initialEndDate,
        initialPointTo,
        setStartDate,
        setEndDate,
        requestSuggests,
        setToPoint,
    ]);

    return (
        <SearchForm
            searchForm={searchForm}
            searchSuggests={suggests.items}
            setToPoint={setToPoint}
            setStartDate={setStartDate}
            setEndDate={setEndDate}
            setAdults={setAdults}
            queryParams={queryParams}
            setChildrenAges={setChildrenAges}
            requestSuggests={manualRequestSuggests}
            submitButtonText={searchSubmitButton()}
            onShowFieldPopup={handleFormInteraction}
            onSuggestInputFocus={handleFormInteraction}
            onSubmit={handleSubmit}
        />
    );
};

export default HotelsSearchFormContainer;
