import {URLs} from 'constants/urls';

import {THotelSlug} from 'types/hotels/hotel/IHotel';
import {IBaseSearchQueryParams} from 'types/hotels/common/IQueryParams';
import {ESearchSuggestItemType} from 'types/hotels/common/ISearchSuggest';
import {IHotelsSearchFormValues} from 'projects/hotels/components/SearchForm/types';

import {internalUrl} from 'utilities/url';
import getISObyDate from 'utilities/dateUtils/getISObyDate';
import {getHotelUrlByHotelSlug} from 'projects/hotels/utilities/urls/getHotelUrl';
import {createLastSearchTimeMarker} from 'projects/hotels/utilities/getLastSearchTimeMarkerQuery';

const getHotelSlugByFields = (
    searchFormFieldsValues: IHotelsSearchFormValues,
): THotelSlug | undefined => {
    const toSelectedValue = searchFormFieldsValues.to.selectedValue;

    if (typeof toSelectedValue === 'boolean') {
        return undefined;
    }

    return toSelectedValue.redirectParams.hotelSlug;
};

const getSearchRouteUrl = ({
    activeHotelSlug,
    searchFormFieldsValues,
    isSearchHotelsPage,
    hotelSlugByRouteMatch,
}: {
    activeHotelSlug?: THotelSlug;
    searchFormFieldsValues: IHotelsSearchFormValues;
    isSearchHotelsPage: boolean;
    hotelSlugByRouteMatch?: THotelSlug;
}): string => {
    if (isSearchHotelsPage) {
        return URLs.hotelsSearch;
    }

    const hotelSlug = getHotelSlugByFields(searchFormFieldsValues);

    if (activeHotelSlug) {
        return getHotelUrlByHotelSlug(activeHotelSlug);
    }

    if (hotelSlug) {
        return getHotelUrlByHotelSlug(hotelSlug);
    }

    return hotelSlugByRouteMatch
        ? getHotelUrlByHotelSlug(hotelSlugByRouteMatch)
        : URLs.hotelsHotelPage;
};

const getSearchPageQueryParams = (
    searchFormFieldsValues: IHotelsSearchFormValues,
) => {
    const toSelectedValue = searchFormFieldsValues.to.selectedValue;

    if (typeof toSelectedValue === 'boolean') {
        return {};
    }

    return {
        geoId: toSelectedValue.redirectParams.geoId,
        selectedSortId: toSelectedValue.redirectParams.selectedSortId,
        sortOrigin: toSelectedValue.redirectParams.sortOrigin,
    };
};

const getBaseSearchQueryParams = (
    searchFormFieldsValues: IHotelsSearchFormValues,
): IBaseSearchQueryParams => {
    const {adults, childrenAges, startDate, endDate} = searchFormFieldsValues;

    return {
        adults,
        childrenAges: childrenAges && childrenAges.join(','),
        checkinDate: startDate ? getISObyDate(new Date(startDate)) : undefined,
        checkoutDate: endDate ? getISObyDate(new Date(endDate)) : undefined,
    };
};

const checkRegionAndSearchGroupType = (
    searchFormFieldsValues: IHotelsSearchFormValues,
    activeHotelSlug?: THotelSlug,
): boolean => {
    const toSelectedValue = searchFormFieldsValues.to.selectedValue;

    if (typeof toSelectedValue === 'boolean') {
        return false;
    }

    const suggestGroupType = toSelectedValue.redirectParams.type;

    if (activeHotelSlug) {
        return false;
    }

    return (
        suggestGroupType === ESearchSuggestItemType.REGION ||
        suggestGroupType === ESearchSuggestItemType.HOTELS_NEARBY ||
        suggestGroupType === ESearchSuggestItemType.CROSS_SALE ||
        suggestGroupType === ESearchSuggestItemType.CROSS_SEARCH ||
        suggestGroupType === ESearchSuggestItemType.HISTORY
    );
};

export const getSearchUrl = ({
    activeHotelSlug,
    searchFormFieldsValues,
    hotelSlugByRouteMatch,
    queryParams,
}: {
    activeHotelSlug?: THotelSlug;
    searchFormFieldsValues: IHotelsSearchFormValues;
    hotelSlugByRouteMatch?: THotelSlug;
    queryParams?: Record<string, string>;
}): string => {
    const isSearchHotelsPage = checkRegionAndSearchGroupType(
        searchFormFieldsValues,
        activeHotelSlug,
    );
    const searchRouteUrl = getSearchRouteUrl({
        activeHotelSlug,
        searchFormFieldsValues,
        hotelSlugByRouteMatch,
        isSearchHotelsPage,
    });

    const baseSearchParams = getBaseSearchQueryParams(searchFormFieldsValues);
    const lastSearchTimeMarkerParams = createLastSearchTimeMarker();
    const searchParamsByType = getSearchPageQueryParams(searchFormFieldsValues);

    return internalUrl(
        searchRouteUrl,
        {
            ...baseSearchParams,
            ...searchParamsByType,
            ...lastSearchTimeMarkerParams,
            ...queryParams,
        },
        {filterNull: true},
    );
};
