import {FunctionComponent, useCallback} from 'react';

import * as i18nBlock from 'i18n/widget-constructor';

import RadioButtonSlide from 'components/RadioButtonSlide/RadioButtonSlide';
import Text from 'components/Text/Text';
import Flex from 'components/Flex/Flex';
import Select from 'components/SelectOld/SelectOld';
import SelectButton from 'components/SelectOld/components/SelectButton/SelectButton';

import {EWidgetViewType} from '../WidgetPreview/WidgetPreview';

import cx from './DesktopTouchSwitcher.scss';

export interface IDesktopTouchSwitcherProps {
    widgetViewType: EWidgetViewType;
    changeWidgetViewType: (value: EWidgetViewType) => void;
    touchSize: EWidgetViewTouchSize;
    changeTouchSize: (value: EWidgetViewTouchSize) => void;
}

export enum EWidgetViewTouchSize {
    S = 's',
    M = 'm',
    L = 'l',
}

const TOUCH_SIZES = [
    {
        text: '320px',
        value: EWidgetViewTouchSize.S,
    },
    {
        text: '550px',
        value: EWidgetViewTouchSize.M,
    },
    {
        text: '768px',
        value: EWidgetViewTouchSize.L,
    },
];

const DesktopTouchSwitcher: FunctionComponent<IDesktopTouchSwitcherProps> = ({
    widgetViewType,
    changeWidgetViewType,
    touchSize,
    changeTouchSize,
}) => {
    const onTouchSizeChange = useCallback(
        ({value}: {value: string | number; type: string}) => {
            if (
                Object.values(EWidgetViewTouchSize).includes(
                    value as EWidgetViewTouchSize,
                )
            ) {
                changeTouchSize(value as EWidgetViewTouchSize);
            }
        },
        [changeTouchSize],
    );

    const getTouchSizeSwitcher = useCallback(() => {
        return (
            <SelectButton className={cx('selectButton')}>
                {TOUCH_SIZES.find(size => size.value === touchSize)?.text}
            </SelectButton>
        );
    }, [touchSize]);

    return (
        <Flex alignItems="center">
            <Text size="s">{i18nBlock.previewLabel()}</Text>
            <RadioButtonSlide
                name="simple"
                value={widgetViewType}
                onChange={changeWidgetViewType}
                size="m"
                className={cx('radioButton')}
            >
                <RadioButtonSlide.Option value={EWidgetViewType.DESKTOP}>
                    <Text size="s">desktop</Text>
                </RadioButtonSlide.Option>
                <RadioButtonSlide.Option value={EWidgetViewType.TOUCH}>
                    <Text size="s">touch</Text>
                </RadioButtonSlide.Option>
            </RadioButtonSlide>
            {widgetViewType === EWidgetViewType.TOUCH && (
                <div className={cx('touchSizeSwitcher')}>
                    <Select
                        items={TOUCH_SIZES}
                        switcherComponent={getTouchSizeSwitcher}
                        onChange={onTouchSizeChange}
                    />
                </div>
            )}
        </Flex>
    );
};

export default DesktopTouchSwitcher;
