import {
    FunctionComponent,
    useCallback,
    useEffect,
    useMemo,
    useReducer,
} from 'react';
import {widgetsApi} from 'api/widgetsApi';

import {
    HOTELS_SUGGEST_DEFAULT_DOMAIN,
    HOTELS_SUGGEST_DEFAULT_LANGUAGE,
    SEARCH_SUGGEST_LIMIT,
} from 'constants/hotels';

import {IHotelsSuggest} from 'types/hotels/common/ISearchSuggest';
import {ESearchFormFieldName} from 'components/SearchForm/types';

import useImmutableCallback from 'utilities/hooks/useImmutableCallback';
import getSuggestTitleAndDescription from 'projects/hotels/components/SearchForm/utilities/getSuggestTitleAndDescription';
import isNearbySuggest from 'components/SearchSuggest/utilities/isNearbySuggest';

import {ISuggestValue} from 'components/SearchSuggest/SearchSuggest';
import useFieldsRefsAndErrors from 'components/SearchForm/hooks/useFieldsRefsAndErrors';
import Suggest from 'components/SearchForm/components/Suggest/Suggest';

import {setHotelsSearchFormToFieldAction} from 'projects/hotels/containers/SearchForm/reducers/searchForm/actions';
import {fetchSearchSuggestActions} from 'projects/hotels/containers/SearchForm/reducers/suggests/actions';
import searchFormReducer, {
    SEARCH_FORM_INITIAL_STATE,
} from 'projects/hotels/containers/SearchForm/reducers/searchForm/reducer';
import suggestsReducer, {
    SUGGESTS_INITIAL_STATE,
} from 'projects/hotels/containers/SearchForm/reducers/suggests/reducer';

import cx from './ToFieldInput.scss';

const ToFieldInput: FunctionComponent<{}> = () => {
    const [suggests, dispatchSuggests] = useReducer(
        suggestsReducer,
        SUGGESTS_INITIAL_STATE,
    );

    const [searchForm, dispatchSearchForm] = useReducer(
        searchFormReducer,
        SEARCH_FORM_INITIAL_STATE,
    );

    const {
        to: toField,
        startDate: startDateField,
        endDate: endDateField,
    } = searchForm;

    const setToPoint = useCallback(
        (fieldValue: ISuggestValue<IHotelsSuggest>) =>
            dispatchSearchForm(setHotelsSearchFormToFieldAction(fieldValue)),
        [],
    );

    const {sessionId, requestIndex} = suggests;

    const requestSuggests = useImmutableCallback(
        async (options: {
            query?: string;
            regionSlug?: string;
            hotelSlug?: string;
        }): Promise<IHotelsSuggest[]> => {
            dispatchSuggests(fetchSearchSuggestActions.request());

            try {
                const {data} = await widgetsApi.searchSuggest({
                    query: options.query,
                    hotelSlug: options.hotelSlug,
                    regionSlug: options.regionSlug,
                    sessionId,
                    requestIndex,
                    limit: SEARCH_SUGGEST_LIMIT,
                    domain: HOTELS_SUGGEST_DEFAULT_DOMAIN,
                    language: HOTELS_SUGGEST_DEFAULT_LANGUAGE,
                });

                const items = data.items.filter(item => !isNearbySuggest(item));

                dispatchSuggests(
                    fetchSearchSuggestActions.success({
                        items,
                        timeMarker: Date.now(),
                    }),
                );

                return items;
            } catch (err) {
                dispatchSuggests(fetchSearchSuggestActions.failure());

                return [];
            }
        },
    );

    const manualRequestSuggests = useImmutableCallback(
        (fieldInputValue: string) => requestSuggests({query: fieldInputValue}),
    );

    useEffect(() => {
        if (toField) {
            manualRequestSuggests(toField.inputValue);
        }
    }, [manualRequestSuggests, toField]);

    const formValues = useMemo(() => {
        return {
            [ESearchFormFieldName.FROM]: undefined,
            [ESearchFormFieldName.TO]: toField,
            [ESearchFormFieldName.START_DATE]: startDateField,
            [ESearchFormFieldName.END_DATE]: endDateField,
        };
    }, [endDateField, startDateField, toField]);

    const {setFieldRefByName, setErrorFieldRefByName} = useFieldsRefsAndErrors({
        fieldsNames: [ESearchFormFieldName.TO],
        formValues,
        canShowErrors: false,
        formErrors: {},
    });

    return (
        <>
            {toField && setToPoint && (
                <Suggest
                    className={cx('suggest')}
                    triggerClassName={cx('suggestTrigger')}
                    searchFormName={'searchFormName'}
                    name={ESearchFormFieldName.TO}
                    nextFieldName={ESearchFormFieldName.START_DATE}
                    value={toField}
                    error={''}
                    otherField={false}
                    placeholder={''}
                    suggestItems={suggests.items}
                    setPoint={setToPoint}
                    triggerFocusClassName={cx('suggestTrigger_focus')}
                    focusNextFieldByName={() => {}}
                    setErrorFieldRefByName={setErrorFieldRefByName}
                    setFieldRefByName={setFieldRefByName}
                    getSuggestTitleAndDescription={
                        getSuggestTitleAndDescription
                    }
                    setPreviousSearchFormValue={() => {}}
                    uniqueValueName={'id'}
                    storeField={() => {}}
                    onInputFocus={() => {}}
                    restoreField={() => {}}
                    onSelectItem={() => {}}
                    previousSuggestItems={undefined}
                />
            )}
        </>
    );
};

export default ToFieldInput;
