import {
    FunctionComponent,
    useCallback,
    useContext,
    useMemo,
    useState,
} from 'react';
import Select from 'widgets/Select/Select';
import EditableSelect from 'widgets/EditableSelect/EditableSelect';

import {EPartner} from 'types/EPartner';
import {EWidgetType} from 'types/widgets/EWidgetType';

import {useBoolean} from 'utilities/hooks/useBoolean';
import getWidgetCode from '../utilities/getWidgetCode';
import copyToClipboard from 'utilities/copyToClipboard';

import * as i18nBlock from 'i18n/widget-constructor';
import * as i18nSearchFormWidgetBlock from 'i18n/widget-search-form';
import * as i18nHintsBlock from 'i18n/widget-constructor-hints';

import Heading from 'components/Heading/Heading';
import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import Button from 'components/Button/Button';
import {ISelectOption} from 'components/SelectOld/SelectOld';

import ConstrutorContext, {
    DEFAULT_REDIRECT_DOMAIN,
    IConstrutorContextParams,
} from 'contexts/ConstrutorContext';

import WidgetConstructorInput from '../WidgetConstructorInput/WidgetConstructorInput';
import ToFieldInput from '../ToFieldInput/ToFieldInput';
import WidgetConstructorControl from '../WidgetConstructorControl/WidgetConstructorControl';

import cx from './WidgetConstructorControlPanel.scss';

export interface IWidgetConstructorControlPanelProps {
    partner?: EPartner;
    onChange: (newContext: IConstrutorContextParams) => void;
}

export const SELET_OPTION_FOR_EDIT: ISelectOption = {
    text: i18nBlock.selectOptionForEditLabel(),
    value: 'editVid',
};

const WidgetConstructorControlPanel: FunctionComponent<
    IWidgetConstructorControlPanelProps
> = ({partner, onChange}) => {
    const {
        value: isShowControls,
        setTrue: showControls,
        setFalse: showScript,
    } = useBoolean(true);

    const value = useContext(ConstrutorContext);
    const {
        clid,
        clids,
        vids,
        vid,
        redirectDomain,
        utmSource,
        utmMedia,
        utmCampaign,
        pointTo,
    } = value || {};

    const onSelectClid = useCallback(
        selectedOption => {
            onChange({...value, clid: selectedOption.value});
        },
        [onChange, value],
    );

    const onInputClidChange = useCallback(
        (inputValue: string) => {
            onChange({...value, clid: inputValue});
        },
        [onChange, value],
    );

    const clidField = useMemo(() => {
        switch (partner) {
            case EPartner.DISTRIBUTION:
                return (
                    <WidgetConstructorControl
                        label="Clid"
                        tooltipText={i18nHintsBlock.clidHint()}
                    >
                        <Select
                            options={
                                clids
                                    ? clids.map((c: string) => ({
                                          text: c,
                                          value: c,
                                      }))
                                    : [{text: clid || '', value: clid || ''}]
                            }
                            selectedOption={{
                                text: clid || '',
                                value: clid || '',
                            }}
                            onSelectOption={onSelectClid}
                        />
                    </WidgetConstructorControl>
                );
            case EPartner.TRAVELPAYOUTS:
                return null;
            case EPartner.ADMITAD:
                return null;
            default:
                return (
                    <WidgetConstructorControl
                        label="Clid"
                        tooltipText={i18nHintsBlock.clidHint()}
                    >
                        <WidgetConstructorInput
                            value={clid}
                            onChange={onInputClidChange}
                        />
                    </WidgetConstructorControl>
                );
        }
    }, [partner, clid, clids, onSelectClid]);

    const onSelectVid = useCallback(
        selectedOption => {
            onChange({...value, vid: selectedOption.value});
        },
        [onChange, value],
    );

    const onInputVidChange = useCallback(
        (inputValue: string) => {
            onChange({...value, vid: inputValue});
        },
        [onChange, value],
    );

    const [vidSelectOptions, setVidSelectOptions] = useState(
        vids?.length
            ? [
                  ...vids.map((c: string) => ({
                      text: c,
                      value: c,
                  })),
                  SELET_OPTION_FOR_EDIT,
              ]
            : [],
    );

    const uptateVidsOptions = useCallback(
        (newOption: ISelectOption) => {
            if (value.vids) {
                onChange({...value, vids: [...value.vids, newOption.text]});
                onSelectVid(newOption);
                setVidSelectOptions([
                    ...[...value.vids, newOption.text].map((c: string) => ({
                        text: c,
                        value: c,
                    })),
                    SELET_OPTION_FOR_EDIT,
                ]);
            }
        },
        [value, onChange, onSelectVid, setVidSelectOptions],
    );

    const vidField = useMemo(() => {
        switch (partner) {
            case EPartner.DISTRIBUTION:
                return vids?.length ? (
                    <WidgetConstructorControl
                        label="Vid"
                        tooltipText={i18nHintsBlock.vidHint()}
                    >
                        <EditableSelect
                            selectedOption={{text: vid || '', value: vid || ''}}
                            options={vidSelectOptions}
                            onSelectOption={onSelectVid}
                            uptateOptions={uptateVidsOptions}
                        />
                    </WidgetConstructorControl>
                ) : (
                    <WidgetConstructorControl
                        label="Vid"
                        tooltipText={i18nHintsBlock.vidHint()}
                    >
                        <WidgetConstructorInput
                            value={vid || ''}
                            onChange={onInputVidChange}
                        />
                    </WidgetConstructorControl>
                );
            case EPartner.TRAVELPAYOUTS:
                return null;
            case EPartner.ADMITAD:
                return null;
            default:
                return (
                    <WidgetConstructorControl
                        label="Vid"
                        tooltipText={i18nHintsBlock.vidHint()}
                    >
                        <WidgetConstructorInput
                            value={vid || ''}
                            onChange={onInputVidChange}
                        />
                    </WidgetConstructorControl>
                );
        }
    }, [partner, vids, vid, onSelectVid, onInputVidChange]);

    const onRedirectDomainInputChange = useCallback(
        (inputValue: string) => {
            onChange({...value, redirectDomain: inputValue});
        },
        [onChange],
    );

    const redirectDomainField = useMemo(() => {
        switch (partner) {
            case EPartner.DISTRIBUTION:
                return redirectDomain !== DEFAULT_REDIRECT_DOMAIN ? (
                    <Text size="s">
                        {i18nBlock.redirectDomainValue({
                            redirectDomain,
                        })}
                    </Text>
                ) : null;
            case EPartner.TRAVELPAYOUTS:
                return null;
            case EPartner.ADMITAD:
                return null;
            default:
                return (
                    <WidgetConstructorControl
                        label={i18nBlock.redirectDomainLabel()}
                        tooltipText={i18nHintsBlock.redirectDomainHint()}
                    >
                        <WidgetConstructorInput
                            value={redirectDomain || ''}
                            onChange={onRedirectDomainInputChange}
                        />
                    </WidgetConstructorControl>
                );
        }
    }, [onRedirectDomainInputChange, redirectDomain]);

    const onUtmSourceInputChange = useCallback(
        (inputValue: string) => {
            onChange({...value, utmSource: inputValue});
        },
        [onChange],
    );

    const onUtmMediaInputChange = useCallback(
        (inputValue: string) => {
            onChange({...value, utmMedia: inputValue});
        },
        [onChange],
    );

    const onutmCampaignInputChange = useCallback(
        (inputValue: string) => {
            onChange({...value, utmCampaign: inputValue});
        },
        [onChange],
    );

    const {code, container} = getWidgetCode({
        partner,
        clid,
        vid,
        redirectDomain,
        utmCampaign,
        utmMedia,
        utmSource,
        pointTo,
    });

    const copyCodeToClipboard = useCallback(() => {
        copyToClipboard({text: code});
    }, [code]);

    return (
        <Flex className={cx('root')} flexDirection="column" between={5}>
            <Heading level={2}>{i18nBlock.title()}</Heading>
            {isShowControls ? (
                <Flex flexDirection="column" between={5}>
                    <WidgetConstructorControl
                        label={i18nBlock.widgetTypeLabel()}
                        tooltipText={i18nHintsBlock.widgetTypeHint()}
                    >
                        <Select
                            selectedOption={{
                                text: i18nSearchFormWidgetBlock.hotelsSearchFormTitle(),
                                value: EWidgetType.HOTELS_SEARCH_FORM,
                            }}
                            onSelectOption={() => {}}
                            options={[
                                {
                                    text: i18nSearchFormWidgetBlock.hotelsSearchFormTitle(),
                                    value: EWidgetType.HOTELS_SEARCH_FORM,
                                },
                            ]}
                        />
                    </WidgetConstructorControl>

                    {clidField}

                    {vidField}

                    {redirectDomainField}

                    {!partner && (
                        <WidgetConstructorControl
                            label="utm_source"
                            tooltipText={i18nHintsBlock.utmSourceHint()}
                        >
                            <WidgetConstructorInput
                                value={utmSource || ''}
                                onChange={onUtmSourceInputChange}
                            />
                        </WidgetConstructorControl>
                    )}

                    {!partner && (
                        <WidgetConstructorControl
                            label="utm_medium"
                            tooltipText={i18nHintsBlock.utmMediumHint()}
                        >
                            <WidgetConstructorInput
                                value={utmMedia || ''}
                                onChange={onUtmMediaInputChange}
                            />
                        </WidgetConstructorControl>
                    )}

                    {!partner && (
                        <WidgetConstructorControl
                            label="utm_campaign"
                            tooltipText={i18nHintsBlock.utmCampaignHint()}
                        >
                            <WidgetConstructorInput
                                value={utmCampaign || ''}
                                onChange={onutmCampaignInputChange}
                            />
                        </WidgetConstructorControl>
                    )}

                    <WidgetConstructorControl
                        label={i18nBlock.whereControlLabel()}
                        tooltipText={i18nHintsBlock.whereHint()}
                    >
                        <ToFieldInput />
                    </WidgetConstructorControl>

                    <Button size="m" theme="primary" onClick={showScript}>
                        {i18nBlock.getCodeButton()}
                    </Button>
                </Flex>
            ) : (
                <Flex flexDirection="column" between={3}>
                    <Flex alignItems="center" justifyContent="space-between">
                        <Text size="m" weight="bold">
                            {i18nBlock.widgetCodeLabel()}
                        </Text>
                        <Button theme="plain" onClick={copyCodeToClipboard}>
                            <Text color="link" size="s">
                                {i18nBlock.copyCodeButton()}
                            </Text>
                        </Button>
                    </Flex>
                    <Text size="s">{i18nBlock.widgetCodeInstruction()}</Text>
                    <pre className={cx('codeFragment')}>
                        <code className={cx('codeContainer')}>{code}</code>
                    </pre>
                    <Text size="m" weight="bold">
                        {i18nBlock.widgetContainerLabel()}
                    </Text>
                    <Text size="s">
                        {i18nBlock.widgetContainerInstruction()}
                    </Text>
                    <pre className={cx('codeFragment')}>
                        <code className={cx('codeContainer')}>{container}</code>
                    </pre>
                    <Button size="m" theme="secondary" onClick={showControls}>
                        {i18nBlock.backButton()}
                    </Button>
                </Flex>
            )}
        </Flex>
    );
};

export default WidgetConstructorControlPanel;
