import {FunctionComponent, useContext, useEffect, useState} from 'react';
import HotelsSearchFormWidget from 'widgets/HotelsSearchForm/HotelsSearchForm';

import {IDevice} from 'types/IDevice';
import {IErrorBooster} from 'types/errorBooster/IErrorBooster';
import {EWidgetType} from 'types/widgets/EWidgetType';
import {EPartner} from 'types/EPartner';

import Metrika from 'utilities/metrika/Metrika';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import Flex from 'components/Flex/Flex';

import ConstrutorContext from 'contexts/ConstrutorContext';

import DesktopTouchSwitcher, {
    EWidgetViewTouchSize,
} from '../DesktopTouchSwitcher/DesktopTouchSwitcher';

import cx from './WidgetPreview.scss';

export enum EWidgetViewType {
    DESKTOP = 'desktop',
    TOUCH = 'touch',
}

export interface IWidgetPreviewProps {
    containerId: string;
    partner?: EPartner;
}

const WidgetPreview: FunctionComponent<IWidgetPreviewProps> = ({
    containerId,
    partner,
}) => {
    const [widgetViewType, setWidgetViewType] = useState<EWidgetViewType>(
        EWidgetViewType.DESKTOP,
    );
    const [touchSize, setTouchSize] = useState<EWidgetViewTouchSize>(
        EWidgetViewTouchSize.S,
    );
    const deviceType = useDeviceType();

    const desktopDeviceType: IDevice = {
        ...deviceType,
        isMobile: false,
        isTouch: false,
        isDesktop: true,
    };
    const touchDeviceType: IDevice = {
        ...deviceType,
        isMobile: true,
        isTouch: true,
        isDesktop: false,
    };

    const value = useContext(ConstrutorContext);
    const {
        clid,
        vid,
        redirectDomain,
        utmSource,
        utmMedia,
        utmCampaign,
        pointTo,
    } = value || {};

    useEffect(() => {
        const hotelsSearchFormWidget = new HotelsSearchFormWidget(
            {
                type: EWidgetType.HOTELS_SEARCH_FORM,
                containerId: `widget_${containerId}`,
                params: {
                    data: {
                        pointTo,
                    },
                },
                affiliate: {
                    affiliateClid: clid,
                    affiliateVid: vid,
                    redirectDomain,
                    utmSource,
                    utmMedia,
                    utmCampaign,
                    partner,
                },
            },
            {
                metrika: new Metrika(),
                deviceType:
                    widgetViewType === EWidgetViewType.DESKTOP
                        ? desktopDeviceType
                        : touchDeviceType,
                errorBooster: {} as IErrorBooster,
            },
        );

        hotelsSearchFormWidget.renderWidget();
    }, [
        widgetViewType,
        containerId,
        partner,
        clid,
        vid,
        redirectDomain,
        utmCampaign,
        utmMedia,
        utmSource,
        pointTo,
    ]);

    return (
        <Flex className={cx('root')} flexDirection="column">
            <DesktopTouchSwitcher
                widgetViewType={widgetViewType}
                changeWidgetViewType={setWidgetViewType}
                touchSize={touchSize}
                changeTouchSize={setTouchSize}
            />
            <div className={cx('widgetPreviewContainer')}>
                <div
                    id={`widget_${containerId}`}
                    className={cx('widgetPreview', {
                        [`widgetPreview_size_${touchSize}`]:
                            widgetViewType === EWidgetViewType.TOUCH,
                    })}
                ></div>
            </div>
        </Flex>
    );
};

export default WidgetPreview;
