import {CHAR_SIX_PER_EM_SPACE} from 'constants/charCodes';

import {ECurrencyType} from 'types/currency/ECurrencyType';
import ValueType from 'types/currency/ValueType';

import formatInteger from 'utilities/numbers/formatInteger';

import * as i18nBlock from 'i18n/common-price';

export const CURRENCY_SYMBOLS = {
    [ECurrencyType.RUB]: '₽',
    [ECurrencyType.RUR]: '₽',
    [ECurrencyType.KZT]: '₸',
    [ECurrencyType.USD]: '$',
    [ECurrencyType.EUR]: '€',

    [ECurrencyType.TRY]: null,
    [ECurrencyType.GBP]: null,
    [ECurrencyType.BYN]: null,
    [ECurrencyType.UAH]: null,
    [ECurrencyType.UZS]: null,
};

const FORMATTERS = {
    eastern: (whole: string, cents: string | undefined): string =>
        typeof cents === 'undefined' ? whole : `${whole},${cents}`,
    western: (whole: string, cents: string | undefined): string =>
        typeof cents === 'undefined' ? whole : `${whole}.${cents}`,
};

const FORMATTERS_WITH_SYMBOL = {
    EASTERN: (
        whole: string,
        cents: string | undefined,
        symbol: string,
        isCurrencyShown: boolean,
        postfix: string | undefined,
    ): string =>
        `${FORMATTERS.eastern(whole, cents)}${postfix || ''}${
            isCurrencyShown ? `${CHAR_SIX_PER_EM_SPACE}${symbol}` : ''
        }`,
    WESTERN: (
        whole: string,
        cents: string | undefined,
        symbol: string,
        isCurrencyShown: boolean,
        postfix: string | undefined,
    ): string =>
        `${isCurrencyShown ? symbol : ''}${FORMATTERS.western(whole, cents)}${
            postfix || ''
        }`,
};

const CURRENCY_FORMATTERS_WITH_SYMBOL = {
    [ECurrencyType.RUB]: FORMATTERS_WITH_SYMBOL.EASTERN,
    [ECurrencyType.RUR]: FORMATTERS_WITH_SYMBOL.EASTERN,
    [ECurrencyType.UAH]: FORMATTERS_WITH_SYMBOL.EASTERN,
    [ECurrencyType.KZT]: FORMATTERS_WITH_SYMBOL.EASTERN,
    [ECurrencyType.BYN]: FORMATTERS_WITH_SYMBOL.EASTERN,
    [ECurrencyType.UZS]: FORMATTERS_WITH_SYMBOL.EASTERN,

    [ECurrencyType.USD]: FORMATTERS_WITH_SYMBOL.WESTERN,
    [ECurrencyType.EUR]: FORMATTERS_WITH_SYMBOL.WESTERN,

    [ECurrencyType.GBP]: null,
    [ECurrencyType.TRY]: null,
};

/**
 * Форматирует значение цены и подставляет симфол валюты.
 *
 * @param value - Значение.
 * @param currency - Код валюты.
 * @param [isCurrencyShown=true] - Показывать знак валюты.
 * @param [isFrom=false] - Показывать префикс от.
 * @param [isRound=false] - Показывать цифры после запятой.
 * @param postfix - Постфикс для value
 * @param thousandsDelimiter - символ разделителя на тысячи
 */
export function formatPrice({
    value,
    currency,
    isCurrencyShown = true,
    isRound = false,
    isFrom = false,
    postfix,
    thousandsDelimiter = CHAR_SIX_PER_EM_SPACE,
}: {
    value: ValueType;
    currency: ECurrencyType;
    isCurrencyShown?: boolean;
    isRound?: boolean;
    isFrom?: boolean;
    postfix?: string;
    thousandsDelimiter?: string;
}): string {
    const symbol = CURRENCY_SYMBOLS[currency] || currency;
    const formatter =
        CURRENCY_FORMATTERS_WITH_SYMBOL[currency] ||
        FORMATTERS_WITH_SYMBOL.EASTERN;
    const preparedValue = typeof value === 'string' ? parseFloat(value) : value;

    let [whole, cents] = preparedValue.toFixed(isRound ? 0 : 2).split('.') as [
        string,
        string | undefined,
    ];

    whole = formatInteger(whole, thousandsDelimiter);

    if (cents === '00') {
        cents = undefined;
    }

    const price = formatter(whole, cents, symbol, isCurrencyShown, postfix);

    if (!isFrom) {
        return price;
    }

    return i18nBlock._from({price});
}
