import moment, {Moment} from 'moment';

import {CHAR_NBSP} from 'constants/charCodes';
import {
    HUMAN,
    HUMAN_SHORT,
    HUMAN_SHORT_WITH_YEAR,
    HUMAN_WITH_YEAR,
} from 'constants/dateFormats';

import {ELocale} from 'types/ELocale';

import localesOverrides from 'utilities/dateUtils/localesOverrides';

/**
 * number - Количество миллисекунд, прошедших с 1 января 1970 года.
 * string - Любая строка, которая преобразуется в дату.
 */

// eslint-disable-next-line @typescript-eslint/naming-convention
export type DateLikeType = number | string | Date | Moment;

export interface IFormatDateRangeParams {
    delimiter?: string;
    forceHideFromDateYearInRange?: boolean;
}

/**
 * Возвращает количество миллисекунд прошедших с 1 января 1970 года до сегодняшнего дня.
 * @return {number}
 */
export function getNow(): number {
    return Date.now();
}

/**
 * Отображение даты в необходимом формате.
 *
 * @param date - Дата, которую нужно преобразовать в строку.
 * @param format - Формат даты.
 * @param options - Настройки форматирования.
 */
export function formatDate(
    date: DateLikeType,
    format: string,
    options?: {
        withNbsp?: boolean;
    },
): string {
    const withNbsp = options?.withNbsp;

    const formattedDate = moment(date).format(format);

    return withNbsp ? formattedDate.replace(/\s/g, CHAR_NBSP) : formattedDate;
}

/**
 * Отображение даты в человекочитаемом виде.
 * Если год даты совпадает с текущим годом, то год не отображается.
 *
 * @param {DateLikeType} date - Дата, которую нужно преобразовать в строку.
 *
 * @return {string}
 */
export function humanFormatDate(date: DateLikeType): string {
    const isCurrentYear = moment(date).isSame(moment(), 'year');
    const yearIsShown = !isCurrentYear;

    return formatDate(date, yearIsShown ? HUMAN_WITH_YEAR : HUMAN);
}

/**
 * Отображение диапазона времени в человекочитаемом виде.
 * Если дата окончания не задана, то будет выведена только дата начала.
 *
 * @param {DateLikeType} from - Дата начала.
 * @param {DateLikeType} [to] - Дата окончания.
 * @param {IFormatDateRangeParams} [params] - Доп параметры.
 *
 * @return {string}
 */
export function formatDateRange(
    from: DateLikeType,
    to?: DateLikeType,
    params?: IFormatDateRangeParams,
): string {
    if (!to) {
        return humanFormatDate(from);
    }

    const {delimiter = ' — ', forceHideFromDateYearInRange} = params || {};
    const fromM = moment(from);
    const toM = moment(to);
    const isSameYear = fromM.isSame(toM, 'year');
    const isCurrentYear = fromM.isSame(moment(), 'year');
    const yearIsShown = !isSameYear || !isCurrentYear;

    const hideFromDate = Boolean(forceHideFromDateYearInRange && to);

    const formattedFrom = formatDate(
        from,
        yearIsShown && !hideFromDate ? HUMAN_SHORT_WITH_YEAR : HUMAN_SHORT,
    );
    const formattedTo = formatDate(
        to,
        yearIsShown ? HUMAN_SHORT_WITH_YEAR : HUMAN_SHORT,
    );

    return `${formattedFrom}${delimiter}${formattedTo}`;
}

/**
 * Перезаписывает настройки локали, если требуется
 *
 * @param {string} locale - язык
 *
 * @return {void}
 */
export function updateLocaleIfNeeded(locale: string): void {
    if (locale === ELocale.RU) {
        moment.updateLocale(locale, localesOverrides[ELocale.RU]);
    }
}

/**
 * Устанавливает язык для дат.
 *
 * @param {string} locale - Язык.
 *
 * @return {string}
 */
export function setDateLocale(locale: string): string {
    updateLocaleIfNeeded(locale);

    return moment.locale(locale);
}
