import axios, {AxiosInstance, AxiosRequestConfig} from 'axios';
import axiosRetry from 'axios-retry';

import {BASE_AXIOS_HEADERS} from 'constants/axios';

import {IHttpClient, IRequestConfig} from 'types/IHttpClient';

interface IBrowserHttpClient {
    baseURL: string;
    timeout?: number;
    httpClientConfig?: AxiosRequestConfig;
}

type TApiMethods = 'GET' | 'DELETE' | 'POST' | 'PUT' | 'PATCH';

export class HttpClient {
    protected httpClient: IHttpClient;

    protected timeout = 25000;

    private readonly axiosInstance: AxiosInstance;

    protected constructor({
        baseURL,
        httpClientConfig = {},
        timeout,
    }: IBrowserHttpClient) {
        this.timeout = timeout || this.timeout;

        const axiosInstance = axios.create({
            baseURL,
            ...httpClientConfig,
        });

        this.axiosInstance = axiosInstance;
        this.httpClient = axiosInstance;
    }

    protected interceptRequest(_request: IRequestConfig): void | Promise<void> {
        return;
    }

    protected get<T>(path: string, options?: IRequestConfig): Promise<T> {
        return this.fetch<T>('GET', path, null, options);
    }

    protected delete<T>(path: string, options?: IRequestConfig): Promise<T> {
        return this.fetch<T>('DELETE', path, null, options);
    }

    protected post<T>(
        path: string,
        body?: unknown,
        options?: IRequestConfig,
    ): Promise<T> {
        return this.fetch<T>('POST', path, body, options);
    }

    protected put<T>(
        path: string,
        body?: unknown,
        options?: IRequestConfig,
    ): Promise<T> {
        return this.fetch<T>('PUT', path, body, options);
    }

    protected patch<T>(
        path: string,
        body?: unknown,
        options?: IRequestConfig,
    ): Promise<T> {
        return this.fetch<T>('PATCH', path, body, options);
    }

    private async fetch<T>(
        method: TApiMethods,
        path: string,
        body?: unknown,
        options?: IRequestConfig,
    ): Promise<T> {
        const requestConfig = this.createRequestConfig(
            path,
            method,
            body,
            options,
        );

        await this.interceptRequest(requestConfig);

        axiosRetry(
            this.axiosInstance,
            requestConfig.axiosRetry || {retries: 0},
        );

        const response = await this.httpClient.request(requestConfig);

        return response.data;
    }

    private createRequestConfig(
        url: string,
        method: TApiMethods,
        body: unknown,
        options?: IRequestConfig,
    ): IRequestConfig {
        const {headers = {}, params = {}, ...requestOptions} = options || {};

        return {
            url,
            method,
            timeout: this.timeout,
            data: body,
            headers: {
                ...BASE_AXIOS_HEADERS,
                ...headers,
            },
            params,
            ...requestOptions,
        };
    }
}
