import axios from 'axios';

interface IVaultVersionValue {
    key: string;
    value: string;
}

interface IVaultSecretVersion {
    comment: string;
    created_at: number;
    created_by: number;
    creator_login: string;
    secret_name: string;
    secret_uuid: string;
    value: IVaultVersionValue[];
    version: string;
}

interface IVaultVersionResponse {
    status: 'ok' | 'error' | 'warning';
    version: IVaultSecretVersion;
}

const YAV_ENDPOINT = 'https://vault-api.passport.yandex.net/1';
const VAULT_AUTH_KEY = process.env.VAULT_AUTH_KEY;

if (!VAULT_AUTH_KEY) {
    console.error('VAULT_AUTH_KEY must be specified in env variables');
    process.exit(1);
}

const yavClient = axios.create({
    baseURL: YAV_ENDPOINT,
    headers: {
        Authorization: VAULT_AUTH_KEY,
    },
});

function getSecretByVersion(version: string): Promise<IVaultVersionResponse> {
    return yavClient.get(`/versions/${version}/`).then(res => res.data);
}

export function setEnvVarsWithSecret(version: string): Promise<void> {
    return getSecretByVersion(version).then(response => {
        if (response.status !== 'ok') {
            return;
        }

        response.version.value.forEach(secret => {
            process.env[secret.key] = secret.value;
        });
    });
}

export function getEnvVarsStringWithSecret(version: string): Promise<string> {
    return getSecretByVersion(version).then(response => {
        if (response.status !== 'ok') {
            return '';
        }

        return response.version.value
            .map(({key, value}) => `${key}=${value}`)
            .join('\n');
    });
}
