const fs = require('fs');
const path = require('path');
const webpack = require('webpack');
const TerserPlugin = require('terser-webpack-plugin');
const HtmlWebpackPlugin = require('html-webpack-plugin');
const MiniCSSExtractPlugin = require('mini-css-extract-plugin');
const {BundleAnalyzerPlugin} = require('webpack-bundle-analyzer');
const CssMinimizerPlugin = require('css-minimizer-webpack-plugin');
const MomentLocalesPlugin = require('moment-locales-webpack-plugin');

const globalVariables = require('./tools/globalVariables');

const isDevelopment = process.env.NODE_ENV !== 'production';

const rootPath = fs.realpathSync(process.cwd());
const srcPath = path.join(rootPath, 'src');
const stylesPath = path.join(srcPath, 'client', 'styles');
const nodeModulesPath = path.join(rootPath, 'node_modules');

module.exports = {
    mode: isDevelopment ? 'development' : 'production',
    devtool: false,
    entry: {
        client: './src/client/index.ts',
        server: './src/server/index.ts',
    },
    module: {
        rules: [
            {
                test: /\.([tj])sx?$/,
                exclude: /node_modules/,
                use: ['babel-loader'],
            },
            {
                test: /\.css$/,
                exclude: /node_modules\/(?!(@yandex-lego)\/).*/,
                use: [MiniCSSExtractPlugin.loader, 'css-loader'],
            },
            {
                test: /\.scss$/,
                exclude: /node_modules\/(?!(@yandex-lego)\/).*/,
                use: [
                    MiniCSSExtractPlugin.loader,
                    {
                        loader: 'css-loader',
                        options: {
                            importLoaders: 4,
                            modules: {
                                localIdentName:
                                    '[folder]__[local]_[hash:base64:5]',
                            },
                        },
                    },
                    {
                        loader: 'resolve-url-loader',
                        options: {
                            sourceMap: true,
                        },
                    },
                    {
                        loader: 'postcss-loader',
                        options: {
                            sourceMap: true,
                        },
                    },
                    {
                        loader: 'sass-loader',
                        options: {
                            implementation: require.resolve('sass'),
                            sourceMap: true,
                            sassOptions: {
                                includePaths: [stylesPath],
                            },
                        },
                    },
                    {
                        loader: 'sass-resources-loader',
                        options: {
                            sourceMap: true,
                            hoistUseStatements: true,
                            resources: [
                                path.join(stylesPath, 'mixins.scss'),
                                path.join(stylesPath, 'variables.scss'),
                                path.join(stylesPath, 'spacing.scss'),
                            ],
                        },
                    },
                ],
            },
            {
                test: /\.(ico|bmp|gif|jpe?g|png|svg|webp)$/,
                include: /node_modules\/@yandex-lego/,
                type: 'asset/resource',
                generator: {
                    filename: '[contenthash][ext]',
                },
            },
        ],
    },
    plugins: [
        new webpack.DefinePlugin(globalVariables),
        isDevelopment &&
            new HtmlWebpackPlugin({
                template: path.resolve(__dirname, 'src/client/index.html'),
                filename: 'index.html',
                meta: {
                    viewport:
                        'width=device-width, initial-scale=1, maximum-scale=1, user-scalable=no',
                },
                chunks: ['client'],
            }),
        new HtmlWebpackPlugin({
            template: path.resolve(__dirname, 'src/server/service.html'),
            filename: 'service.html',
            chunks: ['server'],
        }),
        new MiniCSSExtractPlugin({
            filename: '[name].css',
            ignoreOrder: true,
        }),
        new MomentLocalesPlugin({localesToKeep: ['ru']}),
        process.env.BUILD_BUNDLE_ANALYZER &&
            new BundleAnalyzerPlugin({
                analyzerMode: 'static',
                openAnalyzer: true,
            }),
    ].filter(Boolean),
    resolve: {
        extensions: ['.tsx', '.ts', '.js'],
        alias: {
            types: path.join(srcPath, 'types'),
            icons: path.join(srcPath, 'icons'),
            widgets: path.join(srcPath, 'client', 'widgets'),
            'moment-timezone': path.join(
                nodeModulesPath,
                'moment-timezone',
                'builds',
                'moment-timezone-with-data-2012-2022.min',
            ),
        },
    },
    devServer: {
        client: {
            logging: 'log',
        },
        static: {
            directory: path.join(__dirname, 'dist'),
        },
        proxy: {
            '/api': {
                target: 'https://travel.yandex.ru',
                secure: false,
                changeOrigin: true,
            },
        },
        headers: {
            'X-Custom-Foo': 'bar',
            'Access-Control-Allow-Origin': '*',
        },
        compress: true,
        allowedHosts: ['affiliate.travel-test.yandex.ru'],
        port: process.env.DEV_SERVER_PORT || 9000,
    },
    optimization: {
        minimizer: isDevelopment
            ? []
            : [
                  new TerserPlugin({
                      terserOptions: {
                          parse: {
                              ecma: 8,
                          },
                          compress: {
                              ecma: 5,
                              warnings: false,
                              comparisons: false,
                          },
                          output: {
                              ecma: 5,
                              comments: false,
                          },
                          sourceMap: true,
                      },
                  }),
                  new CssMinimizerPlugin({
                      minimizerOptions: {
                          preset: [
                              'default',
                              {
                                  discardComments: {removeAll: true},
                              },
                          ],
                      },
                  }),
              ],
        splitChunks: {
            chunks: 'all',
        },
        realContentHash: false,
    },
    output: {
        filename: '[name].bundle.js',
        path: path.resolve(__dirname, 'dist'),
        publicPath: isDevelopment
            ? undefined
            : process.env.YENV === 'production'
            ? `https://yastatic.net/s3/travel/widgets/${process.env.BUILD_ID}`
            : `https://s3.mdst.yandex.net/travel-frontend/widgets/${process.env.BUILD_ID}/static/`,
        clean: true,
    },
};
