import { NextApiRequest } from 'next';
import { TTvmResult } from 'yandex-net/core/tvm';

import { ServerConfig } from '@/shared/config/server';
import { YandexCookieName, YandexHeaderName } from 'yandex-net/core/shared';

import { blackBoxClient, tvmClient } from '@/app/server/network';

export const getApiRouteIP = (req: NextApiRequest) =>
    process.env.APPLICATION_ENV === 'local'
        ? ServerConfig.localBlackboxProxyIP
        : (req.headers['x-forwarded-for'] as string);

export async function getApiRouteAuthorization(req: NextApiRequest) {
    const tickets = await tvmClient.getTickets();
    const blackbox = await blackBoxClient.sessionid({
        host: req.headers.host as string,
        userip: getApiRouteIP(req),
        [YandexCookieName.SESSION_ID]: req.cookies[YandexCookieName.SESSION_ID]!,
        [YandexHeaderName.X_YA_SERVICE_TICKET]: tickets.asRecord.blackbox.ticket,
    });

    return {
        tickets: tickets.asRecord,
        blackbox,
    };
}

export const getApiRouteProxyHeaders = (
    req: NextApiRequest,
    { tickets, blackbox }: IApiRouteAuthorization,
) => ({
    [YandexHeaderName.X_YA_SERVICE_TICKET]: tickets.api.ticket,
    [YandexHeaderName.X_YA_USER_TICKET]: blackbox.user_ticket,
    [YandexHeaderName.X_YA_PASSPORT_ID]: blackbox.uid.value,
    [YandexHeaderName.X_YA_YANDEX_UID]: req.cookies[YandexCookieName.YANDEX_UID] as string,
    [YandexHeaderName.X_YA_USER_IP]: getApiRouteIP(req),
    [YandexHeaderName.X_YA_LOGIN]: blackbox.login,
});

export interface IApiRouteAuthorization {
    tickets: TTvmResult<'api' | 'blackbox'>;
    // TODO Типизировать
    blackbox: any;
}
