import { useStore } from 'effector-react';
import { memo, ReactNode } from 'react';
import { isPublishedHotelConnectionState } from '@/entities/hotel';
import { Card, Text, TextSkeleton } from '@yandex-travel/ui/atoms';
import { Label } from '@yandex-travel/ui/molecules';

import * as model from '../model';

export interface HotelViewProps {
    children?: ReactNode;
    afterNode?: ReactNode;
}

export const HotelView = memo(({ children, afterNode }: HotelViewProps) => {
    const {
        slug,
        permalink,
        hotel_name,
        st_ticket,
        billing_client_id,
        billing_contract_id,
        paper_agreement,
        payments_enabled,
        accountant_emails,
        unpublished_reason,
        contract_person_emails,
        hotel_connection_state,
    } = useStore(model.$hotel);
    const pending = useStore(model.$pending);

    if (pending) {
        return (
            <>
                <TextSkeleton type="h1" width={240} className="mb-4" />
                <TextSkeleton type="m" width={180} className="mb-4" />
                <Card>
                    <TextSkeleton type="h3" width={160} className="mb-2" />
                    <TextSkeleton type="m" width={120} />
                </Card>
            </>
        );
    }

    if (!permalink && !slug && !hotel_name) {
        return (
            <Text size="xxl" className="mb-4">
                Не найдено
            </Text>
        );
    }

    return (
        <>
            <Text size="xxl" className="mb-4">
                {hotel_name}
            </Text>
            <div className="flex gap-4 mb-4">
                <a className="link" href={`https://st.yandex-team.ru/${st_ticket}`} target="_blank">
                    Трекер: {st_ticket}
                </a>
                <a
                    className="link"
                    href={`https://travel.yandex.ru/hotels/${slug}`}
                    target="_blank"
                >
                    Страница в Yandex Travel
                </a>
            </div>
            <Card>
                <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4 mb-4">
                    <GridItem label="Пермалинк">{permalink}</GridItem>
                    <GridItem label="ID клиента">{billing_client_id}</GridItem>
                    <GridItem label="ID контракта">{billing_contract_id}</GridItem>
                    <GridItem label="Статус договора">
                        {paper_agreement ? 'Бумага' : 'Договор'}
                    </GridItem>
                    <GridItem label="Статус подключения">
                        <Label
                            size="md"
                            theme={
                                isPublishedHotelConnectionState(hotel_connection_state)
                                    ? 'primary'
                                    : 'attention'
                            }
                        >
                            {hotel_connection_state.toLowerCase()}
                        </Label>
                    </GridItem>
                    <GridItem label="Статус выплат">
                        <Label size="md" theme={payments_enabled ? 'primary' : 'attention'}>
                            {payments_enabled ? 'Включены' : 'Отключены'}
                        </Label>
                    </GridItem>
                    <GridItem label="Причина распубликации">{unpublished_reason}</GridItem>
                    <GridItem label="Рассылочная почта">
                        <div className="space-y-2">
                            {accountant_emails.length === 0 && '-'}
                            {accountant_emails.map(email => (
                                <div key={email}>{email}</div>
                            ))}
                        </div>
                    </GridItem>
                    <GridItem label="Почта contract person">
                        <div className="space-y-2">
                            {contract_person_emails.length === 0 && '-'}
                            {contract_person_emails.map(email => (
                                <div key={email}>{email}</div>
                            ))}
                        </div>
                    </GridItem>
                </div>
                {children}
            </Card>
            {afterNode}
        </>
    );
});

function GridItem({ label, children }: { label: ReactNode; children: ReactNode }) {
    return (
        <div>
            <Text size="lg" className="mb-4">
                {label}
            </Text>
            {children}
        </div>
    );
}
