import { useEvent, useStore, useStoreMap } from 'effector-react';
import { memo } from 'react';
import { useFieldArray, useForm } from 'react-hook-form';
import { Form } from '@/shared/ui/atoms';
import { Field } from '@/shared/ui/molecules';
import clsx from 'clsx';
import { Icon, Text, TextSkeleton } from '@yandex-travel/ui/atoms';
import { useEventCallback } from '@yandex-travel/ui/hooks';
import { Button, Input } from '@yandex-travel/ui/molecules';

import { MoveHotelsOrderParams } from '@/features/move-hotel-orders/types';

import * as model from '../model';

export const MoveHotelOrdersView = memo(() => {
    const form = useForm<MoveHotelsOrderParams>({
        defaultValues: {
            orders: [
                {
                    order_id: '',
                    pretty_id: '',
                },
            ],
        },
    });
    const handleSubmit = useEvent(model.submitted);
    const pending = useStore(model.$pending);

    const orders = useFieldArray({
        control: form.control,
        name: 'orders',
    });
    const handleAddClick = useEventCallback(() =>
        orders.append({
            order_id: '',
            pretty_id: '',
        }),
    );

    return (
        <Form form={form} onSubmit={handleSubmit}>
            <div className="grid grid-cols-2 gap-4">
                <Field name="client_id" label="Новый ID клиента" required>
                    <Input fullWidth disabled={pending} />
                </Field>
                <Field name="contract_id" label="Новый ID контракта" required>
                    <Input fullWidth disabled={pending} />
                </Field>
            </div>
            <Text size="lg" className="mb-4">
                Брони
            </Text>
            <div className="space-y-1">
                {orders.fields.map((field, index) => (
                    <Item
                        key={field.id}
                        index={index}
                        onRemove={orders.fields.length > 1 ? () => orders.remove(index) : undefined}
                    />
                ))}

                <div className="flex space-x-4">
                    <Button
                        theme="secondary"
                        disabled={pending}
                        onClick={handleAddClick}
                        className="mb-4"
                        startNode={<Icon name="Plus" />}
                    >
                        Добавить
                    </Button>
                    <Button type="submit" disabled={pending}>
                        Перенести
                    </Button>
                </div>
            </div>
        </Form>
    );
});

const Item = memo(({ index, onRemove }: { index: number; onRemove?(): void }) => {
    const pending = useStore(model.$pending);
    const result = useStoreMap({
        store: model.$results,
        keys: [index],
        fn: (results, [index]) => results[index] ?? null,
    });

    return (
        <div className="w-full flex gap-4 items-start">
            <Field
                name={`orders.${index}.pretty_id`}
                label="Order Pretty ID"
                required
                className="flex-1"
                description={
                    pending || result ? (
                        <>
                            {pending && !result && (
                                <TextSkeleton type="xs" className="block" width={200} />
                            )}
                            {result && (
                                <span
                                    className={clsx(
                                        result.status === 'fail'
                                            ? 'text-text-alert'
                                            : 'text-text-success',
                                    )}
                                >
                                    {result.status === 'fail'
                                        ? result.error.message
                                        : result.result.move_result || 'Готово'}
                                </span>
                            )}
                        </>
                    ) : null
                }
            >
                <Input fullWidth disabled={pending} placeholder="Например: YA-1111-2222-3333" />
            </Field>
            {onRemove && (
                <Button
                    theme="secondary"
                    className="mt-5.5"
                    disabled={pending}
                    onClick={onRemove}
                    startNode={<Icon name="Delete" />}
                />
            )}
        </div>
    );
});
