const { resolve } = require('path');
const { readFile, writeFile, mkdir, access } = require('fs/promises');

const cacheDir = resolve(__dirname, '_generated');
const getCacheFile = name => resolve(cacheDir, name);

const ensureCacheDir = () => access(cacheDir).catch(() => mkdir(cacheDir));
const readCacheFile = (name, fallback = null) =>
    readFile(name, 'utf-8')
        .then(JSON.parse)
        .catch(() => fallback);
const writeCacheFile = (name, content) =>
    writeFile(name, JSON.stringify(content, null, 2), 'utf-8');

class ArgumentsStore {
    constructor(name, defaults = {}) {
        this.fileName = getCacheFile(`${name}.json`);
        this.defaults = defaults;
    }

    async get() {
        await ensureCacheDir();

        const content = await readCacheFile(this.fileName);

        return {
            ...this.defaults,
            ...content?.value,
        };
    }

    async set(value) {
        await ensureCacheDir();

        return writeCacheFile(this.fileName, { value });
    }
}

module.exports = ArgumentsStore;
