const { readdir, readFile, stat } = require('fs/promises');
const { optimize } = require('svgo');
const { parse } = require('svgson');
const { resolve, join } = require('path');
const { createElement } = require('react');
const { renderToStaticMarkup } = require('react-dom/server');

const renderIconsToSprite = nodes =>
    renderToStaticMarkup(createSpriteElement(nodes.map(createIconElement)));

const createSpriteElement = icons =>
    createElement('svg', { width: 0, height: 0, className: 'hidden' }, icons);

const createIconElement = node =>
    createElementFromNode(
        {
            ...node,
            name: 'symbol',
        },
        node.attributes.id,
    );

/**
 * Каждая поддиректория считается отдельным svg-спрайтом
 */
async function readSvgGroupsInPath(path, parent = '') {
    const children = await readdir(path);

    return Promise.all(
        children.map(async name => {
            const childPath = resolve(path, name);
            const childStat = await stat(childPath);
            const childName = join(parent, name);

            if (childStat.isDirectory()) {
                return readSvgGroupsInPath(childPath, childName);
            }

            if (childStat.isFile()) {
                // Игнорируем рутовую директорию
                if (!parent) return null;

                return { group: parent, info: getSvgFileInfo(name, childPath) };
            }

            return null;
        }),
    ).then(results => [].concat(...results.filter(Boolean)));
}

/**
 * Для генерации спрайта из конкретной директории, без группировки
 */
async function readSvgEntriesInPath(path) {
    const children = await readdir(path);

    return Promise.all(
        children.map(async name => {
            const childStat = await stat(resolve(path, name));

            return childStat.isFile() ? [name] : [];
        }),
    )
        .then(results => [].concat(...results))
        .then(results => results.map(name => getSvgFileInfo(name, resolve(path, name))));
}

/**
 * Обрабатываем svg-файл, получаем на выход оптимизированное ast-дерево
 * @param {string} id
 * @param {string} path
 * @returns {Promise<INode>}
 */
async function processFile(id, path) {
    const content = await readFile(path, 'utf-8');
    const optimized = optimize(content, SVGO_DEFAULTS);

    if (optimized.error) {
        throw new Error(optimized.error);
    }

    const transformNode = node =>
        Array.isArray(node) ? node.map(transformNode) : applySVGNodeTransform(node, { id });

    return await parse(optimized.data, {
        transformNode,
        camelcase: true,
    });
}

module.exports = {
    readSvgGroupsInPath,
    readSvgEntriesInPath,
    processFile,
    createIconElement,
    createSpriteElement,
    renderIconsToSprite,
};

// ==========
// Internal
// ==========

const createElementFromNode = ({ children, name, attributes }, key) =>
    createElement(
        name,
        { ...attributes, key },
        Array.isArray(children)
            ? children.map((node, index) =>
                  node.type === 'element'
                      ? createElementFromNode(node, index)
                      : node.type === 'text'
                      ? node.value
                      : undefined,
              )
            : children,
    );

const getSvgFileInfo = (name, path) => ({
    id: name.replace(/.svg$/, ''),
    name,
    path,
});

const getPatchedColorAttributes = attributes =>
    CONVERTED_ATTRIBUTES.reduce((acc, name) => {
        if (attributes[name] && CONVERTED_COLORS.includes(attributes[name])) {
            acc[name] = 'currentColor';
        }

        return acc;
    }, {});

const applySvgChildNodeTransform = node => ({
    ...node,
    ...applyCommonNodeTransform(node),
    attributes: {
        ...node.attributes,
        ...getPatchedColorAttributes(node.attributes),
    },
});
const applyCommonNodeTransform = node => ({
    children: Array.isArray(node.children)
        ? node.children.map(applySvgChildNodeTransform)
        : node.children,
});
const applySVGNodeTransform = (node, { id }) => ({
    ...node,
    ...applyCommonNodeTransform(node),
    attributes: {
        ...node.attributes,
        id,
    },
});

const SVGO_DEFAULTS = {
    plugins: [
        { name: 'removeStyleElement', active: true },
        { name: 'removeUselessStrokeAndFill', active: true },
        { name: 'removeScriptElement', active: true },
        { name: 'removeEmptyAttrs', active: true },
        { name: 'mergePaths', active: true },
        { name: 'collapseGroups', active: true },
        { name: 'removeTitle', active: true },
        // Первый отключаем - удаляет viewBox.
        // Второй удаляет width/height и гарантирует наличие viewBox
        { name: 'removeViewBox', active: false },
        { name: 'removeDimensions', active: true },
        {
            name: 'removeAttrs',
            params: {
                attrs: [
                    '(class|style)',
                    'xlink:href',
                    'aria-labelledby',
                    'aria-describedby',
                    'xmlns:xlink',
                    'data-name',
                ],
            },
        },
    ],
    multipass: true,
};

const CONVERTED_ATTRIBUTES = ['fill', 'stroke'];
const CONVERTED_COLORS = ['#333333', '#333', '#000000', '#000'];
