const { access, mkdir, writeFile } = require('fs/promises');
const { resolve, dirname } = require('path');

const { processFile, readSvgGroupsInPath, renderIconsToSprite } = require('../lib');

const { prompts } = require('../../../internal');

module.exports = {
    async prompt({ prompter, args, h }) {
        const ora = (await import('ora')).default;
        const { prompt, getAnswers } = prompts(prompter, {
            defaults: {
                name: 'my-icons',
                outputStatic: 'static',
                ...args,
            },
            silent: args.silent,
        });

        const { input, outputRoot, outputStatic } = await prompt([
            {
                name: 'name',
                type: 'input',
                result: h.changeCase.kebab,
                message: 'Имя группы спрайтов',
            },
            {
                name: 'input',
                type: 'input',
                silent: true,
                message: 'Path to folder with svg files',
                validate: path =>
                    access(resolve(path))
                        .then(() => true)
                        .catch(() => `Path "${path}" not exists`),
            },
            {
                name: 'outputRoot',
                type: 'input',
                message: 'Корневая директория для сгенерированных файлов',
            },
            {
                name: 'outputStatic',
                type: 'input',
                message: 'Относительный путь для расположения сгенерированных спрайтов',
            },
        ]);

        const readingFilesProgress = ora(`Looking for svg sources`).start();

        const inputPath = resolve(input);
        const scannedInfo = await readSvgGroupsInPath(inputPath);
        const groups = scannedInfo.reduce((acc, item) => {
            if (!acc[item.group]) {
                acc[item.group] = {
                    name: item.group.replace(/\//g, '.'),
                    path: item.group,
                    items: [],
                    processed: [],
                };
            }

            acc[item.group].items.push(item.info);

            return acc;
        }, {});
        const groupsCount = Object.keys(groups).length;
        const groupsAsList = Object.values(groups);

        readingFilesProgress.succeed(`Found ${scannedInfo.length} files in ${groupsCount} groups`);

        const processingProgress = ora(getProgressLabel(0, scannedInfo.length)).start();
        let groupIndex = 0;
        let index = 0;

        for (const group of groupsAsList) {
            groupIndex++;

            for (const item of group.items) {
                processingProgress.text = getProgressLabel(
                    groupsCount,
                    scannedInfo.length,
                    groupIndex,
                    ++index,
                    group.name,
                );
                group.processed.push(await processFile(item.id, item.path));
            }
        }

        processingProgress.succeed(
            `Processed ${scannedInfo.length} files in ${groupsCount} groups`,
        );

        const generatingSpriteProgress = ora(`[0/${groupsCount}] Generating sprites...`).start();
        const outputRootPath = resolve(outputRoot);
        const outputStaticPath = resolve(outputRootPath, outputStatic);

        groupIndex = 0;

        await ensurePath(outputStaticPath);

        for (const group of groupsAsList) {
            generatingSpriteProgress.text = `[${++groupIndex}/${groupsCount}] Generating sprites...`;
            group.generatedSpritePath = resolve(outputStaticPath, `${group.name}.svg`);
            // console.log(group.processed[0]);
            await writeFile(group.generatedSpritePath, renderIconsToSprite(group.processed));
        }

        generatingSpriteProgress.succeed();
        console.log('Generated sprite files:');
        console.log(groupsAsList.map(group => ` > ${group.generatedSpritePath}`).join('\n'));

        return {
            ...getAnswers(),
            groupsAsList,
        };
    },
};

const getProgressLabel = (groups, files, passedGroups = 0, passedFiles = 0, currentGroup = '') =>
    `[${passedFiles}/${files} files] [${passedGroups}/${groups} groups] Processing "${currentGroup}"...`;

const ensurePath = path =>
    access(path).catch(() => ensurePath(dirname(path)).then(() => mkdir(path)));
