const { access } = require('fs/promises');
const { resolve } = require('path');

const { processFile, readSvgEntriesInPath, renderIconsToSprite } = require('../lib');

const { prompts } = require('../../../internal');

module.exports = {
    async prompt({ prompter, args }) {
        const ora = (await import('ora')).default;
        const { prompt, getAnswers } = prompts(prompter, {
            defaults: {
                name: 'sprite.svg',
                ...args,
            },
            silent: args.silent,
        });

        const { input } = await prompt([
            {
                name: 'input',
                type: 'input',
                silent: true,
                message: 'Path to folder with svg files',
                validate: path =>
                    access(resolve(path))
                        .then(() => true)
                        .catch(() => `Path "${path}" not exists`),
            },
            {
                name: 'output',
                type: 'input',
                message: 'Path to output generated sprites files',
            },
            {
                name: 'name',
                type: 'input',
                message: 'Generated sprite name',
            },
        ]);

        const inputAbsPath = resolve(input);

        const readingFilesProgress = ora(`Looking for svg sources`).start();
        const inputs = await readSvgEntriesInPath(inputAbsPath);

        readingFilesProgress.succeed(`Found ${inputs.length} files`);

        const processingProgress = ora('[0/0] Processing sources').start();
        const processedNodes = [];
        let index = 0;

        for (const { path, name, id } of inputs) {
            processingProgress.text = `[${++index}/${inputs.length}] Processing sources`;
            processedNodes.push({
                id,
                name,
                path,
                value: await processFile(id, path),
            });
        }

        processingProgress.succeed(`Processed ${inputs.length} files`);

        const generatingSpriteProgress = ora('Generating sprite').start();
        const sprite = renderIconsToSprite(processedNodes);

        generatingSpriteProgress.succeed();

        return {
            ...getAnswers(),
            sprite: sprite,
            iconsIds: processedNodes.map(icon => icon.id),
        };
    },
};
