const withBundleAnalyzer = require('@next/bundle-analyzer');
const withPlugins = require('next-compose-plugins');
const withTranspileModules = require('next-transpile-modules');
const log = require('next/dist/build/output/log');

const createEnvFn =
    (fn, { defaultFallback = null }) =>
    (name, { fallback = defaultFallback } = {}) => {
        const value = process.env[name];

        return value !== undefined ? fn(value) : fallback;
    };
const env = {
    string: createEnvFn(p => p, { defaultFallback: '' }),
    number: createEnvFn(Number, { defaultFallback: null }),
    bool: createEnvFn(p => p === 'true', { defaultFallback: false }),
};

module.exports = {
    log,
    env,
    createNextConfig(
        {
            // cwd,
            workspaceDependencies = [],
            analyzer: { enabled, detailed } = {},
            logSettings,
            plugins = [],
        },
        configuration = {},
    ) {
        log.info(
            'Next application settings:\n',
            JSON.stringify(
                {
                    analyzer: {
                        enabled,
                        detailed,
                    },
                    workspaceDependencies,
                    ...logSettings,
                },
                null,
                2,
            ),
        );

        return withPlugins(
            [
                withBundleAnalyzer({
                    enabled,
                }),
                withTranspileModules(workspaceDependencies),
                ...plugins,
            ],
            {
                swcMinify: false,
                reactStrictMode: true,
                ...configuration,
                typescript: {
                    /**
                     * Отключаем проверки на TS при сборке, они будут проходить до этого вместе с остальными проверками
                     */
                    ignoreBuildErrors: env.bool('CI'),
                    ...(configuration.typescript ?? {}),
                },
                experimental: {
                    externalHelpers: true,
                    esmExternals: true,
                    externalDir: false,
                    ...(configuration.experimental ?? {}),
                },
                webpack(config, options) {
                    if (enabled && detailed) {
                        config.optimization.concatenateModules = false;
                        config.optimization.moduleIds = 'named';
                        config.optimization.chunkIds = 'named';
                    }

                    return configuration.webpack ? configuration.webpack(config, options) : config;
                },
            },
        );
    },
};
