const isObject = value => typeof value === 'object' && value !== null;
const MergeStrategy = {
    Shallow: 'shallow',
    Deep: 'deep',
};
const getShallowStrategy = () => MergeStrategy.Shallow;

function shallowMerge(a, b) {
    if (Array.isArray(a) && Array.isArray(b)) {
        return [...a, ...b];
    } else if (isObject(a) && isObject(b)) {
        return { ...a, ...b };
    } else if (typeof b !== 'undefined') {
        return b;
    }

    return a;
}

function deepMerge(a, b, strategy = getShallowStrategy, basePath = null) {
    if (strategy(basePath) === MergeStrategy.Shallow) {
        return shallowMerge(a, b);
    }

    if (Array.isArray(a) && Array.isArray(b)) {
        return shallowMerge(a, b);
    }

    if (isObject(a) && isObject(b)) {
        const keys = [...Object.keys(a), ...Object.keys(b)];
        const final = {};

        for (const key of keys) {
            const path = basePath ? [basePath, key].join('.') : key;

            final[key] = deepMerge(a[key], b[key], strategy, path);
        }

        return final;
    }

    return shallowMerge(a, b);
}

module.exports = {
    MergeStrategy,
    deepMerge,
};
