const { MergeStrategy, deepMerge } = require('./deep-merge');

const getMergeStrategy = path => (path === 'rules' ? MergeStrategy.Shallow : MergeStrategy.Deep);
const mergeConfigReducer = (acc, config) => deepMerge(acc, config, getMergeStrategy);
const parseExtension = pattern => (pattern.startsWith('*.') ? pattern.slice(2) : null);
const toArray = value => (Array.isArray(value) ? value : [value]);

/**
 * Склеиваем конфиг
 */

function splitOverrides(overrides) {
    return overrides.flatMap(({ files, ...config }) =>
        toArray(files).map(file => ({ files: [file], ...config })),
    );
}

function mergeOverridesConfig(overrides) {
    const usedRulesByStar = new Set();
    const usedRulesByExtension = new Map();

    return overrides
        .slice()
        .reverse()
        .flatMap(({ files, rules = {}, ...config }) => {
            const extension = parseExtension(files[0]);

            if (extension === null) {
                return { files, rules, ...config };
            }

            if (extension !== '*' && !usedRulesByExtension.has(extension)) {
                usedRulesByExtension.set(extension, new Set());
            }

            const usedRules =
                extension === '*' ? usedRulesByStar : usedRulesByExtension.get(extension);

            return {
                files,
                rules: Object.entries(rules).reduce((acc, [name, rule]) => {
                    if (!rule || usedRules.has(name)) {
                        return acc;
                    }

                    acc[name] = rule;
                    usedRules.add(rule);

                    return acc;
                }, {}),
                ...config,
            };
        })
        .reverse();
}

function mergeConfig(configs) {
    const config = configs.reduce(mergeConfigReducer, {});

    return {
        ...config,
        overrides: mergeOverridesConfig(splitOverrides(config.overrides ?? [])),
    };
}

module.exports = { mergeConfig };
