const Module = require('node:module');

function getModuleNames(config) {
    const { parser, plugins = [], parserOptions = {}, settings = {} } = config;
    const additionalParser = parserOptions.parser;
    const resolversSettings = settings['import/resolver'] || {};

    return [
        // Опциональный парсер
        parser,
        // Опциональный дополнительный парсер
        additionalParser,
        // Плагины
        ...plugins.map(plugin =>
            plugin.startsWith('@') ? `${plugin}/eslint-plugin` : `eslint-plugin-${plugin}`,
        ),
        // Резолверы импортов
        Object.keys(resolversSettings).map(resolver =>
            resolver.startsWith('eslint-import-resolver')
                ? resolver
                : `eslint-import-resolver-${resolver}`,
        ),
    ]
        .flat()
        .filter(Boolean);
}

function patchRequire(require, resolved) {
    const originalResolve = require.resolve;

    const resolve = function resolve(id, options) {
        for (const { name, path } of resolved) {
            if (id === name) return path;
        }

        return originalResolve(id, options);
    };

    require.resolve = Object.assign(resolve, { paths: originalResolve.paths });
}

function applyModuleResolutionPatch(config) {
    const { ModuleResolver, ConfigArrayFactory } = require('@eslint/eslintrc').Legacy;

    if (ConfigArrayFactory.__patched) return;

    const moduleNames = getModuleNames(config);

    /*
     * Resolve modules relative to eslint-kit
     */

    const resolved = moduleNames.map(name => ({
        name,
        path: require.resolve(name, { paths: [__dirname] }),
    }));

    /*
     * Patch eslint ModuleResolver
     */

    const originalResolve = ModuleResolver.resolve;

    ConfigArrayFactory.__patched = true;

    ModuleResolver.resolve = (id, relativeTo) => {
        for (const { name, path } of resolved) {
            if (id === name) return path;
        }

        return originalResolve(id, relativeTo);
    };

    /*
     * Patch default "require"
     */

    patchRequire(require, resolved);

    /*
     * Patch manually created "require" functions
     */

    const originalCreateRequire = Module.createRequire;

    Module.createRequire = path => {
        const require = originalCreateRequire(path);

        patchRequire(require, resolved);

        return require;
    };

    /*
     * Patch resolver in eslint-module-utils
     */

    const ModuleRequire = require('eslint-module-utils/module-require');

    const originalModuleRequire = ModuleRequire.default;

    ModuleRequire.default = id => {
        for (const { name, path } of resolved) {
            if (id === name) return require(path);
        }

        return originalModuleRequire(id);
    };
}

module.exports = {
    applyModuleResolutionPatch,
};
