const { PresetsNames, conditional } = require('../../lib');

/**
 * Создаёт правила для TS проектов и отключает аналогичные кор правила
 * @param {*} context - контекст кита
 * @param {boolean} enableTypesPrefixes - включает требование interface IFoo/type TBar/enum EBaz
 * @param {boolean} enableRequiredReturn - предупреждение, если у функции отсутствует возвращаемый тип
 * @param {boolean} enableNoUseBeforeDefine - предупреждение, если используемая конструкция декларируется после использования
 * @returns {*}
 */
function createTypescriptRules(
    context,
    {
        enableTypesPrefixes = true,
        enableNoUseBeforeDefine = true,
        enableRequiredReturn = true,
    } = {},
) {
    return {
        '@typescript-eslint/naming-convention': [
            'warn',
            {
                selector: 'variable',
                format: ['camelCase', 'UPPER_CASE', 'PascalCase'],
                leadingUnderscore: 'allow',
            },
            {
                selector: 'parameter',
                format: ['camelCase', 'PascalCase'],
                leadingUnderscore: 'allow',
            },
            {
                selector: 'typeLike',
                format: ['PascalCase'],
            },
            {
                selector: 'enumMember',
                format: ['UPPER_CASE'],
            },
            ...(enableTypesPrefixes
                ? [
                      {
                          selector: 'interface',
                          format: ['PascalCase'],
                          prefix: ['I'],
                      },
                      {
                          selector: 'typeAlias',
                          format: ['PascalCase'],
                          prefix: ['T'],
                      },
                      {
                          selector: 'enum',
                          format: ['PascalCase'],
                          prefix: ['E'],
                      },
                  ]
                : []),
        ],
        '@typescript-eslint/explicit-function-return-type': enableRequiredReturn
            ? [
                  'warn',
                  {
                      allowExpressions: true,
                      allowTypedFunctionExpressions: true,
                      allowDirectConstAssertionInArrowFunctions: true,
                  },
              ]
            : 'off',
        '@typescript-eslint/explicit-member-accessibility': [
            'error',
            { accessibility: 'no-public' },
        ],
        '@typescript-eslint/no-inferrable-types': [
            'error',
            {
                ignoreParameters: true,
                ignoreProperties: true,
            },
        ],
        '@typescript-eslint/type-annotation-spacing': [
            'error',
            {
                before: false,
                after: true,
                overrides: {
                    colon: {
                        before: false,
                        after: true,
                    },
                    arrow: {
                        before: true,
                        after: true,
                    },
                },
            },
        ],
        '@typescript-eslint/no-unused-vars': [
            'warn',
            {
                ignoreRestSiblings: true,
                varsIgnorePattern: '^_',
                argsIgnorePattern: '^_',
            },
        ],
        '@typescript-eslint/no-use-before-define': enableNoUseBeforeDefine
            ? ['warn', { functions: false }]
            : 'off',
        '@typescript-eslint/adjacent-overload-signatures': 'warn',
        '@typescript-eslint/ban-ts-comment': 'error',
        '@typescript-eslint/ban-types': 'error',
        '@typescript-eslint/consistent-type-assertions': 'warn',
        '@typescript-eslint/consistent-type-definitions': ['warn', 'interface'],
        '@typescript-eslint/no-array-constructor': 'warn',
        '@typescript-eslint/no-empty-interface': 'warn',
        '@typescript-eslint/no-explicit-any': [
            'warn',
            {
                // Ломает код, проще руками ставить игнор/фиксить
                fixToUnknown: false,
                ignoreRestArgs: true,
            },
        ],
        '@typescript-eslint/no-extra-non-null-assertion': 'warn',
        '@typescript-eslint/no-for-in-array': 'warn',
        '@typescript-eslint/no-misused-new': 'error',
        '@typescript-eslint/no-namespace': 'error',
        '@typescript-eslint/no-non-null-assertion': 'warn',
        '@typescript-eslint/no-unnecessary-condition': 'warn',
        '@typescript-eslint/no-unused-expressions': 'warn',
        '@typescript-eslint/no-useless-constructor': 'warn',
        '@typescript-eslint/prefer-for-of': 'warn',
        '@typescript-eslint/prefer-includes': 'warn',
        '@typescript-eslint/prefer-namespace-keyword': 'error',
        '@typescript-eslint/prefer-optional-chain': 'warn',
        '@typescript-eslint/prefer-string-starts-ends-with': 'warn',
        '@typescript-eslint/require-array-sort-compare': 'error',
        '@typescript-eslint/return-await': ['error', 'in-try-catch'],
        '@typescript-eslint/triple-slash-reference': 'error',

        // Specific rules that are enabled using @typescript-eslint, but have analogues in common eslint
        camelcase: 'off',
        'no-array-constructor': 'off',
        'no-unused-vars': 'off',
        'no-unused-expressions': 'off',
        'no-use-before-define': 'off',
        'no-useless-constructor': 'off',
        'no-return-await': 'off',

        // Checked by Typescript - ts(2378)
        'getter-return': 'off',
        // Checked by Typescript - ts(2300)
        'no-dupe-args': 'off',
        // Checked by Typescript - ts(1117)
        'no-dupe-keys': 'off',
        // Checked by Typescript - ts(7027)
        'no-unreachable': 'off',
        // Checked by Typescript - ts(2367)
        'valid-typeof': 'off',
        // Checked by Typescript - ts(2588)
        'no-const-assign': 'off',
        // Checked by Typescript - ts(2588)
        'no-new-symbol': 'off',
        // Checked by Typescript - ts(2376)
        'no-this-before-super': 'off',
        // This is checked by Typescript using the option `strictNullChecks`.
        'no-undef': 'off',
        // This is already checked by Typescript.
        'no-dupe-class-members': 'off',
        // This is already checked by Typescript.
        'no-redeclare': 'off',

        ...conditional(context.presets.has(PresetsNames.imports), {
            // Checked by Typescript
            'import/no-extraneous-dependencies': 'off',
        }),
    };
}

module.exports = { createTypescriptRules };
