export interface ObfuscateRecordParams {
    target: Record<string, any>;
    include: string[];
    format?: (value: string) => string;
    formatUnknown?: (value: string) => string;
}

export const formatCookie = (cookie: Record<string, any>) =>
    Object.entries(cookie)
        .map(([key, value]) => [key, encodeURIComponent(value)])
        .join('; ');

export function obfuscateRecord({
    target,
    format = obfuscateByCenter,
    include,
    formatUnknown = identity,
}: ObfuscateRecordParams) {
    return Object.fromEntries(
        Object.entries(target).map(([key, value]) => [
            key,
            typeof value === 'string'
                ? include.includes(key)
                    ? format(value)
                    : formatUnknown(value)
                : value,
        ]),
    );
}

/**
 * @see https://wiki.yandex-team.ru/logfeller/security/
 */

export const obfuscateByLimit = (value: string, limit = 10) =>
    value.length < limit ? value : value.slice(0, limit - 3) + '...';

export const obfuscateByCenter = (value: string) => {
    const center = Math.floor(value.length / 2);

    return value.slice(0, center) + 'X'.repeat(value.length - center);
};

export const obfuscateByLastPart = (value: string, delimiter = '.') =>
    value
        .split(delimiter)
        .map((part, index, parts) =>
            index === parts.length - 1 ? getObfuscated(part.length) : part,
        )
        .join(delimiter);

const getObfuscated = (length: number) => 'X'.repeat(length);
const identity = (value: string) => value;
