import { LoggerFields, LoggerLevel, LoggerMiddleware, LoggerOptions, LoggerTarget } from './types';

import { createChunk } from './lib/create-chunk';
import { parseArguments } from './lib/parse-arguments';

import { JsonStream } from './stream/json';
import { YandexDeployStream } from './stream/yandex-deploy';

export class Logger {
    static level = LoggerLevel;
    static default = new Logger({
        name: 'default-logger',
        streams: [
            {
                level: LoggerLevel.debug,
                stream: typeof window === 'undefined' ? new YandexDeployStream() : new JsonStream(),
            },
        ],
    });

    info = this.createLevel(LoggerLevel.info);
    warn = this.createLevel(LoggerLevel.warn);
    error = this.createLevel(LoggerLevel.error);
    trace = this.createLevel(LoggerLevel.trace);
    fatal = this.createLevel(LoggerLevel.fatal);
    debug = this.createLevel(LoggerLevel.debug);

    readonly fields: LoggerFields;
    private readonly middlewares: LoggerMiddleware[];
    private readonly targets: LoggerTarget[];

    constructor({
        middlewares = [],
        streams = [...Logger.default.targets],
        fields: fieldsAsProp,
        ...otherFields
    }: LoggerOptions) {
        this.fields = {
            ...otherFields,
            ...fieldsAsProp,
        };
        this.targets = streams;
        this.middlewares = middlewares;
    }

    child(
        fields: LoggerFields = {},
        {
            streams = this.targets,
            middlewares = this.middlewares,
        }: Pick<LoggerOptions, 'streams' | 'middlewares'> = {},
    ) {
        return new Logger({
            streams,
            middlewares,
            ...this.fields,
            ...fields,
        });
    }

    update(fields: LoggerFields) {
        Object.assign(this.fields, fields);

        return this;
    }

    private createLevel(level: LoggerLevel) {
        return this.log.bind(this, level);
    }

    private log(level: LoggerLevel, ...args: any[]) {
        const chunk = this.middlewares.reduce(
            (acc, fn) => fn(acc),
            createChunk({
                fields: this.fields,
                parsed: parseArguments(args),
                level,
            }),
        );

        for (const { level: supportedLevel, stream } of this.targets) {
            if (supportedLevel <= level) {
                stream.write(chunk);
            }
        }
    }
}
