import { parse } from 'cookie';

import { LoggerMiddleware } from '../types';

import {
    formatCookie,
    obfuscateByCenter,
    obfuscateByLastPart,
    obfuscateRecord,
} from '../lib/obfuscate';

export function secureHeaders(): LoggerMiddleware {
    return chunk => {
        const headers = chunk.req?.headers;

        if (headers) {
            chunk.req.headers = obfuscateRecord({
                target: headers,
                include: DEFAULT_HEADERS,
                format: obfuscateByCenter,
            });
        }

        return chunk;
    };
}

export function secureCookie(): LoggerMiddleware {
    return chunk => {
        if (!chunk.req) return chunk;

        const { headers, cookies } = chunk.req;

        if (cookies) {
            if (headers) {
                delete headers.cookie;
            }

            chunk.req.cookies = obfuscateCookie(cookies);
        } else if (typeof headers?.cookie === 'string') {
            headers.cookie = formatCookie(obfuscateCookie(parse(headers.cookie)));
        }

        return chunk;
    };
}

const obfuscateCookie = (target: Record<string, any>) =>
    obfuscateRecord({
        target,
        format: obfuscateByLastPart,
        include: DEFAULTS_COOKIE,
    });

const DEFAULT_HEADERS = ['x-ya-user-ticket', 'x-ya-service-ticket', 'ticket', 'authorization'];
const DEFAULTS_COOKIE = [
    'Session_id',
    'Secure_session_id',
    'sessionid2',
    'Eda_id',
    'edaid2',
    'yandex_login',
];
