import { join, resolve } from 'path';
import { OpenAPIV3 } from 'openapi-types';
import { GeneratedClientType } from './render/clients/types';
import { getHelperInfo, HelperName } from './render/helpers';
import { camelCase, pascalCase, STRING_SEPARATOR } from './utils/normalize-string';

export function createConfiguration({
    cwd = process.cwd(),
    generate = ['api', 'sharedTypes', 'endpointsTypes'],
    render: { endpointFullNameSeparator = '/', ...render } = {},
    output: {
        endpointsTypes = 'types/endpoints.ts',
        sharedTypes = 'types/shared.ts',
        contracts = 'contracts.ts',
        effector = 'effector.ts',
        proxy = 'node-proxy.ts',
        redux = 'redux.ts',
        api = 'api.ts',
    } = {},
    resolve: resolveConfig = {},
    sourceRoot = '.',
    outputRoot = 'api/generated',
    helpersRoot = 'api/helpers',
    include,
}: UserConfiguration): Configuration {
    const cwdPath = (relativeToCwd: string): CwdPathInfo => ({
        absolute: resolve(cwd, relativeToCwd),
        relativeToCwd,
    });
    const sourcePath = (relativeToSource: string): SourcePathInfo => ({
        ...cwdPath(join(sourceRoot, relativeToSource)),
        relativeToSource,
    });
    const helperPath = (relativeToHelpers: string): HelperPathInfo => ({
        ...cwdPath(join(helpersRoot, relativeToHelpers)),
        relativeToHelpers,
    });

    return {
        cwd,
        generate,
        render: {
            endpointFullNameSeparator,
            indexedTypeUnknownValue: 'unknown',
            indexedTypeKey: 'string',
            unknownType: 'unknown',
            ...render,
        },
        resolve: {
            schemaName: pascalCase,
            responseName: name => `${pascalCase(name)}Response`,
            parameterName: name => `${pascalCase(name)}Param`,
            requestBodyName: name => `${pascalCase(name)}RequestBody`,
            primaryMediaName: getPrimaryMedia,
            documentName: target => pascalCase(getDefaultDocumentPath(target)),
            documentPath: getDefaultDocumentPath,
            endpointFullName: ctx =>
                ctx.original
                    .operationId!.split(OPERATION_ID_SEPARATOR)
                    .map(pascalCase)
                    .join(endpointFullNameSeparator),
            endpointMethodName: ctx =>
                camelCase(ctx.original.operationId!.split(OPERATION_ID_SEPARATOR).at(-1)!),
            endpointParametersName: ctx => `${pascalCase(ctx.original.operationId!)}Params`,
            endpointSuccessResponseName: ctx => `${pascalCase(ctx.original.operationId!)}Response`,
            isSuccessResponseCode: name => name === '200',
            ...resolveConfig,
        },
        output: {
            ...cwdPath(outputRoot),
            endpointsTypes,
            sharedTypes,
            contracts,
            effector,
            redux,
            proxy,
            api,
        },
        source: {
            ...cwdPath(sourceRoot),
            include: include.map(sourcePath),
        },
        helpers: {
            ...cwdPath(helpersRoot),
            createRequestOptionsFactory: helperPath(
                getHelperInfo('createRequestOptionsFactory').helper.path,
            ),
            createEffectorEndpointEffect: helperPath(
                getHelperInfo('createEffectorEndpointEffect').helper.path,
            ),
            createReduxEndpoint: helperPath(getHelperInfo('createReduxEndpoint').helper.path),
            BaseFetchHttpClient: helperPath(getHelperInfo('BaseFetchHttpClient').helper.path),
            BaseHttpApiClient: helperPath(getHelperInfo('BaseHttpApiClient').helper.path),
            requestFx: helperPath(getHelperInfo('requestFx').helper.path),
        },
    };
}

const getDefaultDocumentPath = ({ info: { title } }: OpenAPIV3.Document) => {
    return title
        .split(DOCUMENT_SEPARATOR)
        .at(-1)!
        .replace('.proto', '')
        .toLowerCase()
        .split(STRING_SEPARATOR)
        .join('-');
};
const getPrimaryMedia = (names: string[]): string | null => {
    if (names.length === 0) {
        return null;
    }

    if (names.length === 1) {
        return names[0];
    }

    return DEFAULT_REQUEST_BODY_MEDIA_PRIORITY.find(name => names.includes(name)) ?? names[0];
};

const DOCUMENT_SEPARATOR = /\//;
const OPERATION_ID_SEPARATOR = /\/|_/;
const DEFAULT_REQUEST_BODY_MEDIA_PRIORITY = [
    'application/json',
    'application/x-www-form-urlencoded',
];

export interface Configuration {
    cwd: string;
    render: RenderConfiguration;
    source: ExtendedSourceConfiguration;
    output: ExtendedOutputConfiguration;
    helpers: ExtendedHelpersConfiguration;
    resolve: ResolveConfiguration;
    generate: GeneratedClientType[];
}

export interface ExtendedSourceConfiguration extends CwdPathInfo {
    include: SourcePathInfo[];
}

export interface ExtendedOutputConfiguration extends CwdPathInfo, OutputConfiguration {}

export interface ExtendedHelpersConfiguration
    extends CwdPathInfo,
        Record<HelperName, HelperPathInfo> {}

export interface OutputPathInfo extends CwdPathInfo {
    relativeToOutput: string;
}

export interface SourcePathInfo extends CwdPathInfo {
    relativeToSource: string;
}

export interface HelperPathInfo extends CwdPathInfo {
    relativeToHelpers: string;
}

export interface CwdPathInfo {
    absolute: string;
    relativeToCwd: string;
}

export interface UserConfiguration {
    /**
     * Корневая директория
     * @default process.cwd()
     */
    cwd?: string;
    /**
     * Настройки рендеринга кода
     */
    render?: Partial<RenderConfiguration>;
    /**
     * Пути к сгенерированным файлам относительно указанной директории
     */
    output?: Partial<OutputConfiguration>;
    /**
     * Настройки обработки openapi-документов
     */
    resolve?: Partial<ResolveConfiguration>;
    generate?: GeneratedClientType[];
    sourceRoot?: string;
    outputRoot?: string;
    helpersRoot?: string;

    /**
     * Список схем
     * @example ["./foo.json", "other/bar.yaml"]
     */
    include: string[];
}

export type OutputConfiguration = Record<GeneratedClientType, string>;

export interface RenderConfiguration {
    /**
     * Используется для формирования полного имени эндпоинта
     *
     * **Не работает при указании своей функции resolve.endpointDisplayName**
     * @example "/"
     * @example "_"
     */
    endpointFullNameSeparator: string;
    unknownType: 'unknown';
    indexedTypeKey: 'string' | 'number' | 'string | number';
    indexedTypeUnknownValue: 'unknown' | 'any';
}

export interface ResolveConfiguration {
    schemaName(name: string): string;
    responseName(name: string): string;
    parameterName(name: string): string;
    requestBodyName(name: string): string;
    primaryMediaName(names: string[]): string | null;
    documentName(target: OpenAPIV3.Document): string;
    documentPath(target: OpenAPIV3.Document): string;
    endpointFullName(context: EndpointResolvingContext): string;
    endpointMethodName(context: EndpointResolvingContext): string;
    endpointParametersName(context: EndpointResolvingContext): string;
    endpointSuccessResponseName(context: EndpointResolvingContext): string;
    isSuccessResponseCode(name: string): boolean;
}

export interface EndpointResolvingContext {
    path: string;
    method: OpenAPIV3.HttpMethods;
    original: OpenAPIV3.OperationObject;
}
