import type { Configuration } from './configuration';
import type { LoadedDocument } from './fs/read';

/**
 * TODO Внедрить везде
 * TODO Добавить детальную (достаточно информативную) статистику
 * Прокидывается во всю внутрянку, чтобы иметь комплексное понимание о процессе генерации
 */
export function createExecutionContext(configuration: Configuration): ExecutionContext {
    let phase: ExecutionPhase = ExecutionPhase.IDLE;
    const scopes = {
        root: createScope(),
        phases: {
            [ExecutionPhase.IDLE]: createScope(),
        } as Partial<Record<ExecutionPhase, Scope>>,
    };
    const computing: Computing[] = [];

    function print() {
        scopes.phases[ExecutionPhase.DONE]?.close();
        scopes.root.close();

        const scopesTimingsLines = PHASES_ORDER.map(
            phase => `${PHASES_LABELS[phase]}: ${scopes.phases[phase]?.toString() ?? '-'}`,
        ).map(line => `  > ${line}`);

        if (phase !== ExecutionPhase.DONE) {
            console.warn(`Неожиданное завершение работы, текущий этап - ${PHASES_LABELS[phase]}`);
        }

        console.info(`Время выполнения: ${scopes.root.toString()}`);
        console.info(scopesTimingsLines.join('\n'));
    }

    return {
        configuration,
        print,
        catch(e: unknown) {
            console.debug('Fatal error - unknown exception');
            print();

            console.log('Configuration:', JSON.stringify(configuration, null, 2));

            if (phase === ExecutionPhase.COMPUTING) {
                const current = computing[computing.length - 1];

                console.debug('Current document:');
                console.log(current);
            }

            console.error(e);
            process.exit(1);
        },
        getPhase: () => phase,
        setPhase(next: ExecutionPhase) {
            scopes.phases[phase]?.close();
            scopes.phases[next] = createScope();
            phase = next;
        },
        setComputingDocument(target: LoadedDocument) {
            computing.push({ target });
        },
    };
}

type Scope = ReturnType<typeof createScope>;

function createScope() {
    const startedAt = Date.now();
    let time = 0;

    return {
        close() {
            time = Date.now() - startedAt;
        },
        time() {
            return time;
        },
        toString() {
            return `${time}ms`;
        },
    };
}

interface Computing {
    target: LoadedDocument;
}

export interface ExecutionContext {
    readonly configuration: Readonly<Configuration>;
    setComputingDocument(source: LoadedDocument): void;
    getPhase(): ExecutionPhase;
    setPhase(phase: ExecutionPhase): void;
    catch(e: unknown): void;
    print(): void;
}

export enum ExecutionPhase {
    IDLE,
    READ,
    COMPUTING,
    RENDERING,
    GENERATING,
    DONE,
}

const PHASES_ORDER: ExecutionPhase[] = [
    ExecutionPhase.IDLE,
    ExecutionPhase.READ,
    ExecutionPhase.COMPUTING,
    ExecutionPhase.RENDERING,
    ExecutionPhase.GENERATING,
    ExecutionPhase.DONE,
];
const PHASES_LABELS: Record<ExecutionPhase, string> = {
    [ExecutionPhase.IDLE]: 'Ожидание',
    [ExecutionPhase.READ]: 'Чтение',
    [ExecutionPhase.COMPUTING]: 'Вычисление',
    [ExecutionPhase.RENDERING]: 'Рендеринг',
    [ExecutionPhase.GENERATING]: 'Запись',
    [ExecutionPhase.DONE]: 'Завершение',
};
