import { access, mkdir, readFile, writeFile } from 'fs/promises';
import { dirname, resolve } from 'path';
import { format, resolveConfig } from 'prettier';
import { ExecutionContext, ExecutionPhase } from '../execution-context';
import { getRequiredClients, getRequiredHelpers, renderClient } from '../render/clients';
import { GeneratedClientType } from '../render/clients/types';
import { HelperInfo } from '../render/helpers';
import { DocumentNode } from '../tree/types';
import { uniq } from '../utils/uniq';

export async function writeDocuments(nodes: RenderedDocument[], context: ExecutionContext) {
    context.setPhase(ExecutionPhase.GENERATING);

    for (const documentNode of nodes) {
        await writeDocument(documentNode, context);
    }

    const helpers = uniq(getRequiredHelpers(getRequiredClients(context.configuration.generate)));

    await writeHelpers(helpers, context);

    context.setPhase(ExecutionPhase.DONE);
}

export function renderDocuments(
    nodes: DocumentNode[],
    context: ExecutionContext,
): RenderedDocument[] {
    context.setPhase(ExecutionPhase.RENDERING);

    return nodes.map(node => renderDocument(node, context));
}

export function renderDocument(node: DocumentNode, context: ExecutionContext): RenderedDocument {
    const clients = getRequiredClients(context.configuration.generate);

    return {
        node,
        generated: Object.fromEntries(
            clients.map(type => [type, renderClient(type, node, context)]),
        ),
    };
}

export async function writeDocument(
    { node, generated }: RenderedDocument,
    context: ExecutionContext,
) {
    await Promise.all(
        context.configuration.generate.map(type =>
            writeOutFile(node.info.output[type].absolute, generated[type]!),
        ),
    );
}

export interface RenderedDocument {
    node: DocumentNode;
    generated: Partial<Record<GeneratedClientType, string>>;
}

async function writeHelpers(helpers: HelperInfo[], context: ExecutionContext) {
    await Promise.all(
        helpers.map(async ({ name, helper: { path, type } }) => {
            const targetPath = context.configuration.helpers[name].absolute;
            const sourcePath = resolve(__dirname, '../src/helpers', path);
            // TODO Добавить ворнинг при обраружении изменений
            const content = await readFile(sourcePath, 'utf-8');

            if (type === 'builtin' || !(await exists(targetPath))) {
                await writeOutFile(targetPath, templates.autogeneratedFile(content));
            }
        }),
    );
}

async function writeOutFile(path: string, content: string) {
    await ensureDir(dirname(path));

    return writeFile(
        path,
        await formatContent(path, templates.autogeneratedFile(content)),
        'utf-8',
    );
}

async function formatContent(path: string, content: string) {
    const config = await resolveConfig(path);

    return format(content, {
        ...config,
        parser: 'typescript',
        filepath: path,
    });
}

// TODO Упростить
const pendingDirs: Record<string, Promise<any>> = {};
const ensureDir = (path: string): Promise<void> => {
    if (!pendingDirs[path]) {
        pendingDirs[path] = access(path)
            .catch(() => ensureDir(dirname(path)).then(() => mkdir(path)))
            .finally(() => {
                delete pendingDirs[path];
            });
    }

    return pendingDirs[path];
};
const exists = (path: string) =>
    access(path)
        .then(() => true)
        .catch(() => false);

const templates = {
    autogeneratedFile: (content: string) => `
/* eslint-disable */
/* tslint:disable */
/*
 * ---------------------------------------------------------------
 * ## THIS FILE WAS GENERATED VIA OPENAPI GENERATOR             ##
 * ##                                                           ##
 * ## Не подлежит изменению, содержимое в любой момент          ##
 * ## может быть автоматически заменено                         ##
 * ---------------------------------------------------------------
 */

 ${content}`,
};
