import { readFile } from 'fs/promises';
import { load } from 'js-yaml';
import { OpenAPIV2, OpenAPIV3 } from 'openapi-types';
// @ts-expect-error no types
import swagger2openapi from 'swagger2openapi';
import { SourcePathInfo } from '../configuration';
import { ExecutionContext, ExecutionPhase } from '../execution-context';

export async function loadDocuments(context: ExecutionContext): Promise<LoadedDocument[]> {
    context.setPhase(ExecutionPhase.READ);

    return Promise.all(
        context.configuration.source.include.map(async source => ({
            source,
            content: await loadDocument(source.absolute),
        })),
    );
}

async function loadDocument(path: string): Promise<OpenAPIV3.Document> {
    const content = await readFile(path, 'utf-8');
    const original = parseStringToDocument(content);

    original.info = Object.assign(
        {
            title: 'Title not provided',
            version: '',
        },
        original.info,
    );

    if (isV3Document(original)) {
        return original;
    }

    original.paths ??= {};

    return new Promise((resolve, reject) => {
        swagger2openapi.convertObj(
            original,
            {
                warnOnly: true,
                refSiblings: 'preserve',
                rbname: 'requestBodyName',
            },
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            (err: any, options: any) => {
                const schema = (err?.options?.openapi ?? options?.openapi) as OpenAPIV3.Document;

                if (!schema && err) {
                    reject(err);
                }

                resolve(schema);
            },
        );
    });
}

function parseStringToDocument(value: string): AnyDocument {
    try {
        return JSON.parse(value);
    } catch (error) {
        try {
            return load(value) as AnyDocument;
        } catch (error) {
            throw new Error(
                `OpenAPI - File parsing error - ${
                    error instanceof Error ? error.message : (error as string).toString()
                }`,
            );
        }
    }
}

const isV3Document = (value: AnyDocument): value is OpenAPIV3.Document =>
    Object.hasOwn(value, 'openapi');

type AnyDocument = OpenAPIV3.Document | OpenAPIV2.Document;

export interface LoadedDocument {
    content: OpenAPIV3.Document;
    source: SourcePathInfo;
}
