import { Action } from 'redux';

export function createReduxEndpoint<Name extends string, Params, Done, Fail = unknown>(name: Name) {
    const actionTypes = createReduxEndpointActionTypes<Name>(name);
    const actions = createReduxEndpointActions<Name, Params, Done, Fail>(actionTypes);
    const initialState: EndpointState<Done, Fail> = {
        data: null,
        error: null,
        status: 'idle',
    };
    const reducer = createReduxEndpointReducer<Name, Params, Done, Fail>(actionTypes);

    return { actions, actionTypes, reducer, initialState };
}

export function createReduxEndpointReducer<Name extends string, Params, Done, Fail>(
    actions: EndpointActionTypes<Name>,
) {
    const nameByActionType = Object.fromEntries(
        Object.entries(actions).map(([name, type]) => [type, name]),
    );
    const reducers: EndpointReducers<Name, Params, Done, Fail> = {
        request: state => ({
            ...state,
            status: 'loading',
        }),
        success: (state, action) => ({
            ...state,
            data: action.payload,
            status: 'success',
        }),
        failure: (state, action) => ({
            ...state,
            error: action.payload,
            status: 'failure',
        }),
    };
    const combined: EndpointCombinedReducer<Name, Params, Done, Fail> = (state, action) => {
        if (nameByActionType[action.type]) {
            return {
                ...state,
                ...reducers[nameByActionType[action.type] as keyof typeof reducers](
                    state,
                    action as any,
                ),
            };
        }

        return state;
    };

    return combined;
}

export function createReduxEndpointActionTypes<Name extends string>(
    name: Name,
): EndpointActionTypes<Name> {
    return {
        request: `${name}/request`,
        success: `${name}/request/success`,
        failure: `${name}/request/failure`,
    };
}

export function createReduxEndpointActions<Name extends string, Params, Done, Fail>(
    types: EndpointActionTypes<Name>,
): EndpointActions<Name, Params, Done, Fail> {
    return {
        request: createReduxActionCreator(types.request),
        success: createReduxActionCreator(types.success),
        failure: createReduxActionCreator(types.failure),
    };
}

export function createReduxActionCreator<Name extends string, Payload>(
    type: Name,
): EndpointActionCreator<Name, Payload> {
    function createAction(payload: Payload): EndpointAction<Name, Payload> {
        return {
            type,
            payload,
        };
    }

    return Object.assign(createAction, {
        action: type,
    });
}

export interface EndpointActionTypes<Name extends string> {
    request: EndpointRequestActionType<Name>;
    success: EndpointSuccessActionType<Name>;
    failure: EndpointFailureActionType<Name>;
}

export interface EndpointActions<Name extends string, Params, Done, Fail> {
    request: EndpointActionCreator<EndpointRequestActionType<Name>, Params>;
    success: EndpointActionCreator<EndpointSuccessActionType<Name>, Done>;
    failure: EndpointActionCreator<EndpointFailureActionType<Name>, Fail>;
}

export interface EndpointReducers<Name extends string, Params, Done, Fail> {
    request(
        state: EndpointState<Done, Fail>,
        action: EndpointRequestAction<Name, Params>,
    ): EndpointState<Done, Fail>;
    success(
        state: EndpointState<Done, Fail>,
        action: EndpointSuccessAction<Name, Done>,
    ): EndpointState<Done, Fail>;
    failure(
        state: EndpointState<Done, Fail>,
        action: EndpointFailureAction<Name, Fail>,
    ): EndpointState<Done, Fail>;
}

export interface EndpointCombinedReducer<Name extends string, Params, Done, Fail> {
    <S extends EndpointState<Done, Fail>>(
        state: S,
        action: EndpointAnyAction<Name, Params, Done, Fail>,
    ): S;
}

export interface EndpointActionCreator<Name extends string, Payload> {
    action: Name;
    (payload: Payload): EndpointAction<Name, Payload>;
}

export interface EndpointAction<Name extends string, Payload> extends Action<Name> {
    payload: Payload;
}

export interface EndpointState<Done, Fail> {
    status: 'idle' | 'loading' | 'success' | 'failure';
    error: Fail | null;
    data: Done | null;
}

export type EndpointAnyAction<Name extends string, Params, Done, Fail> =
    | EndpointRequestAction<Name, Params>
    | EndpointSuccessAction<Name, Done>
    | EndpointFailureAction<Name, Fail>;

export type EndpointRequestAction<Name extends string, Params> = EndpointAction<
    EndpointRequestActionType<Name>,
    Params
>;
export type EndpointSuccessAction<Name extends string, Done> = EndpointAction<
    EndpointSuccessActionType<Name>,
    Done
>;
export type EndpointFailureAction<Name extends string, Fail> = EndpointAction<
    EndpointFailureActionType<Name>,
    Fail
>;

export type EndpointRequestActionType<Name extends string> = `${Name}/request`;
export type EndpointSuccessActionType<Name extends string> = `${Name}/request/success`;
export type EndpointFailureActionType<Name extends string> = `${Name}/request/failure`;
