import { ExecutionContext } from '../../execution-context';
import { DocumentNode, EndpointNode } from '../../tree/types';
import { relativeTsFile } from '../../utils/normalize-string';
import { uniq } from '../../utils/uniq';
import { getHelpersListInfo, renderHelpersImport } from '../helpers';
import { renderEndpointJsDoc } from '../jsdoc';
import { renderImport } from '../shared';
import {
    renderTypeDefinitionsToString,
    renderTypeDefinitionToContentString,
} from '../type-definition/type-definition-to-string';
import { TypeDefinition } from '../type-definition/types';
import { getEndpointFailedResponseTypes, renderEndpointRequestOptionsFactory } from './shared';

export function renderApiClient(node: DocumentNode, context: ExecutionContext) {
    const importPath = (path: string) => relativeTsFile(node.info.output.effector.absolute, path);

    const name = `${node.info.fullName}Api`;
    const methods = node.endpoints.flatMap(path =>
        path.endpoints.map(renderEndpointToMethodString),
    );
    const imports = uniq(methods.flatMap(method => method.imports));
    const content = methods.flatMap(method => method.content).join('\n\n');
    const helpers = getHelpersListInfo(['BaseHttpApiClient', 'createRequestOptionsFactory']);

    const definition: TypeDefinition = {
        content: `
class ${name} extends BaseHttpApiClient {
  ${content}
}
`,
        shouldExportInPublicApi: true,
        type: 'runtime',
        name,
    };

    return [
        renderHelpersImport(node.info.output.api.absolute, helpers, context),
        renderImport(imports.join(','), importPath(node.info.output.endpointsTypes.absolute)),
        renderTypeDefinitionsToString([definition]),
    ].join('\n\n');
}

function renderEndpointToMethodString(endpointNode: EndpointNode) {
    const failed = getEndpointFailedResponseTypes(endpointNode);
    const content = `${endpointNode.display.methodName} = this.createHttpEndpoint<
        ${endpointNode.display.parametersName},
        ${endpointNode.display.responseName},
        ${failed.join(' | ')}
    >(
        ${renderEndpointRequestOptionsFactory(endpointNode)}
    );`;

    return {
        content: renderTypeDefinitionToContentString({
            content,
            name: endpointNode.display.methodName,
            annotation: renderEndpointJsDoc(endpointNode),
            type: 'runtime',
            shouldExportInPublicApi: false,
        }),
        imports: [
            endpointNode.display.responseName,
            endpointNode.display.parametersName,
            ...failed,
        ],
    };
}
