import { ExecutionContext } from '../../execution-context';
import { DocumentNode, EndpointNode } from '../../tree/types';
import { relativeTsFile } from '../../utils/normalize-string';
import { uniq } from '../../utils/uniq';
import { getHelpersListInfo, renderHelpersImport } from '../helpers';
import { renderEndpointJsDoc } from '../jsdoc';
import { renderImport } from '../shared';
import { renderTypeDefinitionsToString } from '../type-definition/type-definition-to-string';
import { TypeDefinition } from '../type-definition/types';
import { getEndpointFailedResponseTypes, renderEndpointRequestOptionsFactory } from './shared';

export function renderEffector(documentNode: DocumentNode, context: ExecutionContext) {
    const importPath = (path: string) =>
        relativeTsFile(documentNode.info.output.effector.absolute, path);

    const effects = documentNode.endpoints
        .flatMap(path => path.endpoints)
        .flatMap(node => renderEndpointToEffectFactory(node, documentNode));
    const endpointTypesImports = uniq(effects.flatMap(effect => effect.importedEndpointTypes));
    const helpers = getHelpersListInfo([
        'createEffectorEndpointEffect',
        'createRequestOptionsFactory',
        'requestFx',
    ]);

    const domain = getDocumentDomainName(documentNode);
    const domainString = `export const ${domain} = createDomain();`;

    return [
        `import { createDomain } from 'effector';`,
        renderHelpersImport(documentNode.info.output.effector.absolute, helpers, context),
        renderImport(
            endpointTypesImports.join(',\n'),
            importPath(documentNode.info.output.endpointsTypes.absolute),
        ),
        domainString,
        renderTypeDefinitionsToString(effects.map(effect => effect.definition)),
    ].join('\n\n');
}

export function renderEndpointToEffectFactory(endpoint: EndpointNode, doc: DocumentNode) {
    const name = `${endpoint.display.methodName}Fx`;
    const failed = getEndpointFailedResponseTypes(endpoint);
    const domain = getDocumentDomainName(doc);

    const content = `
const ${endpoint.display.methodName}Fx = createEffectorEndpointEffect<
    ${endpoint.display.parametersName},
    ${endpoint.display.responseName},
    ${failed.join(' | ')}
>(
    "${endpoint.display.methodName}",
    ${domain},
    requestFx,
    ${renderEndpointRequestOptionsFactory(endpoint)}
);`;

    const definition: TypeDefinition = {
        name,
        type: 'runtime',
        content,
        annotation: renderEndpointJsDoc(endpoint),
    };

    return {
        definition,
        importedEndpointTypes: [
            endpoint.display.parametersName,
            endpoint.display.responseName,
            ...failed,
        ],
    };
}

const getDocumentDomainName = (doc: DocumentNode) => `${doc.info.fullName}Domain`;
