import { ExecutionContext } from '../../execution-context';
import { traverseNodeReferences } from '../../tree/reference';
import { DocumentNode } from '../../tree/types';
import { relativeTsFile } from '../../utils/normalize-string';
import { uniq } from '../../utils/uniq';
import { renderImport, renderJoinedParts, renderRegion } from '../shared';
import {
    renderEndpointParamsDefinition,
    renderEndpointResponseDefinition,
} from '../type-definition/endpoint-to-definition';
import {
    renderTypeDefinitionsToExportString,
    renderTypeDefinitionToContentString,
} from '../type-definition/type-definition-to-string';

export function renderEndpointsTypes(
    { endpoints, resolved, info }: DocumentNode,
    context: ExecutionContext,
) {
    const endpointsGroups = endpoints.map(path => ({
        path,
        items: path.endpoints.map(endpoint => ({
            params: renderEndpointParamsDefinition(endpoint, context.configuration),
            response: renderEndpointResponseDefinition(endpoint, resolved, context.configuration),
            endpoint,
        })),
    }));

    const references = endpointsGroups.flatMap(path =>
        // TODO Собирать только используемые ссылки
        path.items.flatMap(({ endpoint }) => traverseNodeReferences(endpoint, resolved, true)),
    );

    const contentString = renderJoinedParts(
        endpointsGroups.map(({ path, items }) =>
            renderRegion(
                path.path,
                items.flatMap(({ params, response }) => [
                    renderTypeDefinitionToContentString(params),
                    ...response.map(renderTypeDefinitionToContentString),
                ]),
            ),
        ),
    );
    const exportString = renderTypeDefinitionsToExportString(
        endpointsGroups
            .flatMap(group => group.items.flatMap(({ params, response }) => [params, ...response]))
            .filter(Boolean),
    );

    return [
        renderImport(
            uniq(references.map(ref => ref.name)).join(','),
            relativeTsFile(
                info.output.endpointsTypes.relativeToCwd,
                info.output.sharedTypes.relativeToCwd,
            ),
        ),
        contentString,
        exportString,
    ]
        .filter(Boolean)
        .join('\n\n');
}
