import { ExecutionContext } from '../../execution-context';
import { DocumentNode, EndpointNode } from '../../tree/types';
import { relativeTsFile } from '../../utils/normalize-string';
import { uniq } from '../../utils/uniq';
import { getHelpersListInfo, renderHelpersImport } from '../helpers';
import { renderEndpointJsDoc } from '../jsdoc';
import { renderImport } from '../shared';
import { renderTypeDefinitionsToString } from '../type-definition/type-definition-to-string';
import { TypeDefinition } from '../type-definition/types';
import { getEndpointFailedResponseTypes } from './shared';

export function renderReduxClient(documentNode: DocumentNode, context: ExecutionContext) {
    const importPath = (path: string) =>
        relativeTsFile(documentNode.info.output.effector.absolute, path);

    const effects = documentNode.endpoints
        .flatMap(path => path.endpoints)
        .flatMap(renderEndpointToRedux);
    const endpointTypesImports = uniq(effects.flatMap(effect => effect.importedEndpointTypes));
    const helpers = getHelpersListInfo(['createReduxEndpoint']);

    return [
        renderHelpersImport(documentNode.info.output.effector.absolute, helpers, context),
        renderImport(
            endpointTypesImports.join(',\n'),
            importPath(documentNode.info.output.endpointsTypes.absolute),
        ),
        renderTypeDefinitionsToString(effects.map(effect => effect.definition)),
    ].join('\n\n');
}

export function renderEndpointToRedux(endpoint: EndpointNode) {
    const name = `${endpoint.display.methodName}ReduxState`;
    const failed = getEndpointFailedResponseTypes(endpoint);

    const content = `
const ${name} = createReduxEndpoint<
    "${endpoint.display.fullName}",
    ${endpoint.display.parametersName},
    ${endpoint.display.responseName},
    ${failed.join(' | ')}
>("${endpoint.display.fullName}");`;

    const definition: TypeDefinition = {
        name,
        type: 'runtime',
        content,
        annotation: renderEndpointJsDoc(endpoint),
    };

    return {
        definition,
        importedEndpointTypes: [
            endpoint.display.parametersName,
            endpoint.display.responseName,
            ...failed,
        ],
    };
}
