import { ExecutionContext } from '../../execution-context';
import type { DocumentNode, ReferenceSchemaNode } from '../../tree/types';
import { renderSchemaTypeDefinition } from '../type-definition/schema-to-definition';
import { renderTypeDefinitionsToString } from '../type-definition/type-definition-to-string';

export function renderSharedTypes(
    { components: { schemas, parameters, requestBodies, responses } }: DocumentNode,
    context: ExecutionContext,
) {
    const schemasDefinitions = mapObjectEntries(schemas, (name, value) =>
        renderSchemaTypeDefinition(name, value, context.configuration),
    );
    /**
     * Генерим схемы для переиспользуемых параметров, тел запросов и ответов
     * @example interface FooParam { name: string }
     * @example type KekResponse = SharedResponse;
     */
    const parametersDefinitions = mapObjectEntries(parameters, (name, value) =>
        renderSchemaTypeDefinition(
            name,
            value.nodeType === 'ref'
                ? (value as any as ReferenceSchemaNode)
                : value.virtualSchemaNode,
            context.configuration,
        ),
    );
    const requestBodiesDefinitions = mapObjectEntries(requestBodies, (name, value) =>
        renderSchemaTypeDefinition(
            name,
            value.nodeType === 'ref'
                ? (value as any as ReferenceSchemaNode)
                : value.virtualSchemaNode,
            context.configuration,
        ),
    );
    const responsesDefinitions = mapObjectEntries(responses, (name, value) =>
        renderSchemaTypeDefinition(
            name,
            value.nodeType === 'ref'
                ? (value as any as ReferenceSchemaNode)
                : value.virtualSchemaNode,
            context.configuration,
        ),
    );

    return renderTypeDefinitionsToString([
        ...schemasDefinitions,
        ...responsesDefinitions,
        ...parametersDefinitions,
        ...requestBodiesDefinitions,
    ]);
}

function mapObjectEntries<T, R>(target: Record<string, T>, fn: (name: string, value: T) => R): R[] {
    return Object.entries(target).map(([name, value]) => fn(name, value));
}
