import { ExecutionContext } from '../execution-context';
import { relativeTsFile } from '../utils/normalize-string';
import { uniqByT } from '../utils/uniq';
import { renderImport } from './shared';

export const renderHelpersImport = (
    absoluteFromPath: string,
    helpers: HelperInfo[],
    context: ExecutionContext,
) =>
    helpers
        .map(({ name }) =>
            renderImport(
                name,
                relativeTsFile(absoluteFromPath, context.configuration.helpers[name].absolute),
            ),
        )
        .join('\n');

export const getHelpersListInfo = (names: HelperName[]): HelperInfo[] => names.map(getHelperInfo);

export const findAllHelpersRequirements = (names: HelperName[]): HelperInfo[] =>
    uniqHelpers(names.flatMap(findHelperRequirements));

export const getHelperInfo = (name: HelperName): HelperInfo => ({
    name,
    helper: HelpersMap[name],
});

const findHelperRequirements = (name: HelperName): HelperInfo[] =>
    uniqHelpers(
        [getHelperInfo(name)].concat(HelpersMap[name].dependsOn.flatMap(findHelperRequirements)),
    );

const uniqHelpers = uniqByT<HelperInfo>(helper => helper.name);

const HelpersMap: Record<HelperName, HelperDefinition> = {
    createRequestOptionsFactory: {
        path: 'builtin/create-request-options-factory.ts',
        type: 'builtin',
        dependsOn: [],
    },
    BaseFetchHttpClient: {
        path: 'builtin/base-http-client.ts',
        type: 'builtin',
        dependsOn: ['createRequestOptionsFactory'],
    },
    BaseHttpApiClient: {
        path: 'builtin/base-http-api-client.ts',
        type: 'builtin',
        dependsOn: ['BaseFetchHttpClient'],
    },
    createEffectorEndpointEffect: {
        path: 'builtin/create-effector-endpoint-effect.ts',
        type: 'builtin',
        dependsOn: ['createRequestOptionsFactory'],
    },
    createReduxEndpoint: {
        path: 'builtin/create-redux-endpoint.ts',
        type: 'builtin',
        dependsOn: [],
    },
    requestFx: {
        path: 'effector-request-fx.ts',
        type: 'editable',
        dependsOn: ['createRequestOptionsFactory'],
    },
};

export type HelperName =
    | 'createRequestOptionsFactory'
    | 'createEffectorEndpointEffect'
    | 'createReduxEndpoint'
    | 'BaseFetchHttpClient'
    | 'BaseHttpApiClient'
    | 'requestFx';

export interface HelperDefinition {
    path: string;
    type: 'builtin' | 'editable';
    dependsOn: HelperName[];
}

export interface HelperInfo {
    name: HelperName;
    helper: HelperDefinition;
}
