import { getResolved } from '../tree/reference';
import {
    AnySchemaNode,
    EndpointNode,
    ParameterNode,
    RequestBodyNode,
    ResponseNode,
    SchemaNode,
} from '../tree/types';

export function renderSchemaJsDoc(node: AnySchemaNode) {
    if (node.nodeType === 'ref') return null;

    const { example, deprecated } = node.original;
    const description = renderSchemaDescription(node);

    if (!example && !deprecated && description.length === 0) {
        return null;
    }

    return wrapJsDoc(
        renderInnerContent([
            [description.length > 0, description],
            [deprecated, renderDeprecated(deprecated)],
            [example, renderExample(example)],
        ]),
    );
}

export function renderEndpointJsDoc({
    path,
    method,
    responses,
    original: { description, operationId, summary, deprecated },
}: EndpointNode) {
    return wrapJsDoc(
        renderInnerContent([
            [operationId, [operationId || '', '']],
            [summary, [summary || '', '']],
            [description, [`@description ${description}`, '']],
            [deprecated, renderDeprecated(deprecated)],
            [
                true,
                [
                    `@request ${method.toUpperCase()}:${path}`,
                    ...responses.all.map(
                        node =>
                            `@response \`${node.name}\` ${
                                node.value.nodeType === 'ref' ? '`' + node.value.name + '` ' : ''
                            }${getResolved(node.value).original.description}`,
                    ),
                ],
            ],
        ]),
    );
}

export function renderParameterJsDoc({
    original: { description, deprecated, example },
}: ParameterNode) {
    if (!deprecated && !description && !example) return null;

    return wrapJsDoc(
        renderInnerContent([
            [description, splitOptionalMultilineString(description)],
            [deprecated, renderDeprecated(deprecated)],
            [example, renderExample(example)],
        ]),
    );
}

export function renderRequestBodyJsDoc({ original: { description } }: RequestBodyNode) {
    if (!description) return null;

    return wrapJsDoc(
        renderInnerContent([[description, splitOptionalMultilineString(description)]]),
    );
}

export function renderResponseJsDoc({ original: { description } }: ResponseNode) {
    if (!description) return null;

    return wrapJsDoc(
        renderInnerContent([[description, splitOptionalMultilineString(description)]]),
    );
}

function renderSchemaDescription({
    original: { title, description, format, maxLength, minLength, minimum, maximum, pattern },
}: SchemaNode) {
    return prepareInnerContent([
        [title, [...splitOptionalMultilineString(title), ``]],
        [description, splitOptionalMultilineString(description)],
        [format, `@format ${format}`],
        [minimum, `@min ${minimum}`],
        [maximum, `@max ${maximum}`],
        [pattern, `@pattern ${pattern}`],
        [maxLength, `- maxLength: ${maxLength}`],
        [minLength, `- minLength: ${minLength}`],
    ]);
}

const splitOptionalMultilineString = (value?: string) =>
    value?.split('\n').map(chunk => chunk.trim()) ?? [];
const renderDeprecated = (deprecated?: unknown) =>
    `@deprecated ${normalize(String(deprecated ?? '')) || ''}`;
const renderExample = (value?: unknown) => {
    const exampleString = value ? String(value) : '';

    return [
        `@example ${exampleString === '[object Object]' ? JSON.stringify(value) : exampleString}`,
    ];
};

const wrapJsDoc = (value: string) =>
    value
        ? `/**
${value}
 **/`
        : '';
const normalize = (value?: string) => value?.replace(/\*\//g, '*\\/');
const prepareInnerContent = (entries: InnerContentEntry[]) =>
    entries.filter(([value]) => value !== null && value !== undefined).flatMap(entry => entry[1]);
const renderInnerContent = (entries: InnerContentEntry[]) =>
    prepareInnerContent(entries)
        .map(line => ` * ${line}`)
        .join(`\n`);

type InnerContentEntry = [InnerContentKey, InnerContentValue];
type InnerContentKey = string | null | undefined | number | boolean;
type InnerContentValue = string | string[];
