import { Configuration } from '../../configuration';
import { getOrResolveReference } from '../../tree/reference';
import {
    AnyResponseNode,
    DocumentNodeResolved,
    EndpointNode,
    EndpointResponseNode,
} from '../../tree/types';
import { pascalCase } from '../../utils/normalize-string';
import { renderSchemaTypeDefinition } from './schema-to-definition';
import { TypeDefinition } from './types';

export function renderEndpointParamsDefinition(
    node: EndpointNode,
    configuration: Configuration,
): TypeDefinition {
    return renderSchemaTypeDefinition(
        node.display.parametersName,
        node.virtualParametersSchema,
        configuration,
    );
}

export function renderEndpointResponseDefinition(
    node: EndpointNode,
    resolved: DocumentNodeResolved,
    configuration: Configuration,
): TypeDefinition[] {
    const success = node.responses.success
        .map(response =>
            renderAnyResponseNodeDefinition(
                getResponseVariantName(node, response),
                response.value,
                resolved,
                configuration,
            ),
        )
        .filter(Boolean) as TypeDefinition[];
    const failed = node.responses.failed
        .map(response =>
            renderAnyResponseNodeDefinition(
                getResponseVariantName(node, response),
                response.value,
                resolved,
                configuration,
            ),
        )
        .filter(Boolean) as TypeDefinition[];

    const successResponse = renderSchemaTypeDefinition(
        node.display.responseName,
        {
            nodeType: 'schema',
            schemaType: 'object',
            original: {},
            index: null,
            properties: [],
            combined: [
                {
                    type: 'oneOf',
                    value: success.map(type => ({
                        nodeType: 'ref',
                        original: { $ref: '' },
                        name: type.name,
                        type: 'schemas',
                    })),
                },
            ],
        },
        configuration,
    );

    return [successResponse, ...success, ...failed];
}

function renderAnyResponseNodeDefinition(
    name: string,
    node: AnyResponseNode,
    { responses }: DocumentNodeResolved,
    configuration: Configuration,
) {
    const resolved = getOrResolveReference(node, responses);

    return renderSchemaTypeDefinition(
        name,
        resolved.media.primary
            ? resolved.media.primary.schema
            : {
                  nodeType: 'schema',
                  schemaType: 'object',
                  properties: [],
                  combined: null,
                  index: null,
                  original: {},
              },
        configuration,
    );
}

// TODO Вынести в конфигурацию
const getResponseVariantName = (node: EndpointNode, response: EndpointResponseNode) =>
    `${node.display.responseName}${pascalCase(response.name)}`;
