import { Configuration } from '../../configuration';
import { AnySchemaNode, EnumSchemaProperty, ObjectSchemaNode } from '../../tree/types';
import { renderSchemaJsDoc } from '../jsdoc';
import { renderSchemaObjectTypeValue, renderSchemaTypeValue } from './schema-to-value';
import { TypeDefinition } from './types';

export function renderSchemaTypeDefinition(
    name: string,
    node: AnySchemaNode,
    configuration: Configuration,
): TypeDefinition {
    return {
        name,
        type: node.nodeType === 'schema' && node.schemaType === 'enum' ? 'runtime' : 'type',
        content: renderSchemaTypeDefinitionContent(name, node, configuration),
        annotation: renderSchemaJsDoc(node),
        // Реэкспортируем все типы, возможно, потом настроим точечное поведение
        shouldExportInPublicApi: true,
    };
}

/**
 * - TODO Добавить JsDoc по ссылкам (например - поля объектов)
 */
export function renderSchemaTypeDefinitionContent(
    name: string,
    node: AnySchemaNode,
    configuration: Configuration,
) {
    if (node.nodeType === 'ref') {
        /**
         * Для ссылок просто делаем алиас
         * @example type Foo = Bar;
         */
        return templates.type(name, node.name);
    }

    if (!node.combined) {
        if (node.schemaType === 'enum') {
            /**
             * Енамы
             * @example enum Foo { Kek = "Pek" }
             */
            return templates.enum.definition(
                name,
                node.properties.map(templates.enum.value).join(',\n'),
            );
        }

        if (node.schemaType === 'object' && !isEmptyObjectAst(node)) {
            /**
             * Интерфейс объекта без комбинирования типов
             * @example interface Foo { bar: number; baz?: string; }
             */
            return templates.interface(name, renderSchemaObjectTypeValue(node, configuration));
        }
    }

    return templates.type(name, renderSchemaTypeValue(node, configuration));
}

const templates = {
    interface(name: string, value: string) {
        return `interface ${name} ${value}`;
    },
    type(name: string, value: string) {
        return `type ${name} = ${value};`;
    },
    enum: {
        definition(name: string, value: string) {
            return `enum ${name} { ${value} }`;
        },
        value({ value, name }: EnumSchemaProperty) {
            return `${name} = ${value}`;
        },
    },
};

const isEmptyObjectAst = (ast: ObjectSchemaNode) => ast.properties.length === 0 && !ast.index;
