import { OpenAPIV3 } from 'openapi-types';
import { Configuration } from '../../configuration';
import {
    AnySchemaNode,
    ObjectPropertyNode,
    ObjectSchemaIndexType,
    ObjectSchemaNode,
    SchemaCombinedType,
} from '../../tree/types';
import { renderSchemaJsDoc } from '../jsdoc';
import { renderBrackets } from '../shared';

export function renderSchemaTypeValue(node: AnySchemaNode, configuration: Configuration) {
    const content = renderUnknownTypeSelfContent(node, configuration);
    const combined =
        node.nodeType === 'schema' && node.combined
            ? node.combined.map(({ type, value }) => renderCombined(type, value, configuration))
            : [];

    const chunks = [content, ...combined].filter(Boolean) as string[];
    const wrappedChunks = chunks.length > 1 ? chunks.map(renderBrackets) : chunks;

    return wrappedChunks.join('&');
}

const renderCombined = (
    type: SchemaCombinedType,
    nodes: AnySchemaNode[],
    configuration: Configuration,
): string =>
    nodes
        .map(ast => renderSchemaTypeValue(ast, configuration))
        .filter(Boolean)
        .join(mergedSymbols[type]);

/**
 * Строка с определением собственного типа схемы
 * @example string
 * @example { foo: Bar }
 * @example Array<({ name: string }) | (number) | MyType>
 */
function renderUnknownTypeSelfContent(
    node: AnySchemaNode,
    configuration: Configuration,
): string | null {
    if (node.nodeType === 'schema') {
        switch (node.schemaType) {
            case 'enum':
                return node.properties.map(prop => prop.value).join(' | ');
            case 'array':
                return node.value
                    ? `Array<${renderSchemaTypeValue(node.value, configuration)}>`
                    : `Array<${configuration.render.unknownType}>`;
            case 'object':
                return renderSchemaObjectTypeValue(node, configuration);
            default:
                return Object.hasOwn(typeScriptTypesMap, node.schemaType)
                    ? typeScriptTypesMap[node.schemaType as keyof typeof typeScriptTypesMap]
                    : configuration.render.unknownType;
        }
    }

    return node.name;
}

/**
 * Строка со всеми именованными и динамическими свойствами
 * @example {} - пустой объект
 * @example { [key: string]: unknown; } - только динамические
 * @example { foo: string; bar?: Bar; } - только именованные
 * @example { foo: string; bar?: Bar; [key: string]: Baz; } - именованные + динамические
 */
export const renderSchemaObjectTypeValue = (
    { index, properties }: ObjectSchemaNode,
    configuration: Configuration,
) => `{
${[
    properties.length > 0 ? renderObjectProperties(properties, configuration) : null,
    index ? renderObjectAdditionalProperties(index, configuration) : null,
]
    .filter(Boolean)
    .join('\n')}
}`;

/**
 * Дополнительные динамические поля
 */
const renderObjectAdditionalProperties = (
    type: ObjectSchemaIndexType,
    configuration: Configuration,
) =>
    `[key: ${configuration.render.indexedTypeKey}]: ${
        type === 'unknown'
            ? configuration.render.indexedTypeUnknownValue
            : renderSchemaTypeValue(type, configuration)
    } | undefined;`;

/**
 * Строка со всеми именованными свойствами
 */
const renderObjectProperties = (props: ObjectPropertyNode[], configuration: Configuration) =>
    Array.from(props)
        .sort(propertiesComparator)
        .map(prop => renderAnnotatedObjectProperty(prop, configuration))
        .join('\n');

/**
 * Именованное свойство, аннотированное JsDoc
 */
const renderAnnotatedObjectProperty = (prop: ObjectPropertyNode, configuration: Configuration) =>
    [renderSchemaJsDoc(prop.value), renderObjectProperty(prop, configuration)]
        .filter(Boolean)
        .join('\n');

/**
 * Именованное свойство
 * @example foo?: Foo;
 */
const renderObjectProperty = ({ name, value }: ObjectPropertyNode, configuration: Configuration) =>
    `"${name}"${value.original ? '?' : ''}: ${renderSchemaTypeValue(value, configuration)};`;

const propertiesComparator = (left: ObjectPropertyNode, right: ObjectPropertyNode) => {
    if (!left.optional && right.optional) {
        return -1;
    } else if (left.optional && !right.optional) {
        return 1;
    }

    return isAscending(left.name, right.name);
};
const isAscending = (a: string, b: string) => (a > b ? 1 : b > a ? -1 : 0);

const typeScriptTypesMap: Record<Exclude<OpenAPIV3.NonArraySchemaObjectType, 'object'>, string> = {
    integer: 'number',
    number: 'number',
    boolean: 'boolean',
    string: 'string',
};
const mergedSymbols = {
    anyOf: ' | ',
    oneOf: ' | ',
    allOf: ' & ',
};
