import { join, resolve } from 'path';
import type { Configuration, OutputPathInfo } from '../configuration';
import { type ExecutionContext, ExecutionPhase } from '../execution-context';
import type { LoadedDocument } from '../fs/read';
import { createEndpointsPathNode } from './endpoints';
import { createParameterNode } from './parameters';
import { resolveTopLevelReferences, traverseNodeReferences } from './reference';
import { createRequestBodyNode } from './request-bodies';
import { createResponseNode } from './responses';
import { createAnySchemaNode } from './schemas';
import type {
    ComponentType,
    ComponentTypeToNode,
    ComponentTypeToObject,
    DocumentNode,
    ReferenceNode,
} from './types';

export function createDocumentsNodes(
    source: LoadedDocument[],
    context: ExecutionContext,
): DocumentNode[] {
    context.setPhase(ExecutionPhase.COMPUTING);

    return source.map(doc => createDocumentNode(doc, context));
}

export function createDocumentNode(
    target: LoadedDocument,
    context: ExecutionContext,
): DocumentNode {
    context.setComputingDocument(target);

    const { source, content: original } = target;
    const {
        configuration: {
            output,
            resolve: { documentName, documentPath },
        },
    } = context;

    // TODO Добавить в контекст информацию о парсинге
    const schemas = createSchemasMap(original.components?.schemas, context.configuration);
    const responses = createResponsesMap(original.components?.responses, context.configuration);
    const parameters = createParametersMap(original.components?.parameters, context.configuration);
    const requestBodies = createRequestBodiesMap(
        original.components?.requestBodies,
        context.configuration,
    );

    const components = {
        schemas,
        responses,
        parameters,
        requestBodies,
    };
    // TODO Переработать подход к работе со ссылками - сначала собирать элементы без ссылок, а уже потом проходиться по ссылкам
    const resolved = {
        schemas: resolveTopLevelReferences(schemas),
        responses: resolveTopLevelReferences(responses),
        parameters: resolveTopLevelReferences(parameters),
        requestBodies: resolveTopLevelReferences(requestBodies),
    };
    const endpoints = Object.entries(original.paths)
        .filter(entry => Boolean(entry[1]))
        .map(([path, value]) => createEndpointsPathNode(path, value!, context.configuration));

    const fullName = documentName(original);
    const pathName = documentPath(original);
    const outputPath = (path: string): OutputPathInfo => ({
        relativeToOutput: join(pathName, path),
        relativeToCwd: join(output.relativeToCwd, pathName, path),
        absolute: resolve(output.absolute, pathName, path),
    });

    endpoints
        .flatMap(node => node.endpoints)
        .reduce(
            (acc, node) => traverseNodeReferences(node, resolved, false, acc),
            [] as ReferenceNode[],
        );

    return {
        info: {
            source,
            output: {
                endpointsTypes: outputPath(output.endpointsTypes),
                sharedTypes: outputPath(output.sharedTypes),
                contracts: outputPath(output.contracts),
                effector: outputPath(output.effector),
                proxy: outputPath(output.proxy),
                redux: outputPath(output.redux),
                api: outputPath(output.api),
            },
            fullName,
        },
        nodeType: 'document',
        original,
        resolved,
        endpoints,
        components,
    };
}

function createDocumentComponentsMapFactory<Type extends ComponentType>(
    nodeFactory: (
        value: ComponentTypeToObject[Type],
        configuration: Configuration,
    ) => ComponentTypeToNode[Type] | ReferenceNode<Type>,
    resolveName: (name: string, configuration: Configuration) => string,
) {
    return function componentsMapFactory(
        record: Record<string, ComponentTypeToObject[Type]> = {},
        configuration: Configuration,
    ): Record<string, ComponentTypeToNode[Type] | ReferenceNode<Type>> {
        return Object.fromEntries(
            Object.entries(record).map(([name, value]) => [
                resolveName(name, configuration),
                nodeFactory(value, configuration),
            ]),
        );
    };
}

const createSchemasMap = createDocumentComponentsMapFactory(
    createAnySchemaNode,
    (name, configuration) => configuration.resolve.schemaName(name),
);

const createResponsesMap = createDocumentComponentsMapFactory(
    createResponseNode,
    (name, configuration) => configuration.resolve.responseName(name),
);

const createParametersMap = createDocumentComponentsMapFactory(
    createParameterNode,
    (name, configuration) => configuration.resolve.parameterName(name),
);

const createRequestBodiesMap = createDocumentComponentsMapFactory(
    createRequestBodyNode,
    (name, configuration) => configuration.resolve.requestBodyName(name),
);
