import { OpenAPIV3 } from 'openapi-types';
import { Configuration } from '../configuration';
import { createParameterNode } from './parameters';
import { isReferenceNode } from './reference';
import { createRequestBodyNode } from './request-bodies';
import { createResponseNode } from './responses';
import {
    AnyResponseNode,
    EndpointNode,
    EndpointResponseNode,
    EndpointResponsesNode,
    EndpointsPathNode,
    ObjectPropertyNode,
    ObjectSchemaNode,
    ParameterNode,
    ReferenceNode,
    RequestBodyNode,
} from './types';

export function createEndpointsPathNode(
    path: string,
    original: OpenAPIV3.PathItemObject,
    configuration: Configuration,
): EndpointsPathNode {
    const methods = SUPPORTED_HTTP_METHODS.filter(method => Object.hasOwn(original, method));
    const endpoints = methods.map(method =>
        createEndpointNode(path, method, original[method]!, configuration),
    );

    return {
        path,
        original,
        nodeType: 'endpoints-path',
        endpoints,
    };
}

export function createEndpointNode(
    path: string,
    method: OpenAPIV3.HttpMethods,
    original: OpenAPIV3.OperationObject,
    configuration: Configuration,
): EndpointNode {
    const parameters =
        original.parameters?.map(parameter => createParameterNode(parameter, configuration)) ?? [];
    const requestBody = original.requestBody
        ? createRequestBodyNode(original.requestBody, configuration)
        : null;

    const combined = parameters
        .filter<ReferenceNode<any>>(isReferenceNode)
        .concat(requestBody && isReferenceNode(requestBody) ? [requestBody] : []);

    const virtualParametersSchema: ObjectSchemaNode = {
        nodeType: 'schema',
        schemaType: 'object',
        original: {},
        index: null,
        combined:
            combined.length > 0
                ? [
                      {
                          type: 'allOf',
                          value: combined,
                      },
                  ]
                : null,
        properties: parameters
            .filter((node): node is ParameterNode => node.nodeType === 'parameter')
            .map(createObjectPropertyFromParameter)
            .concat(
                requestBody?.nodeType === 'request-body'
                    ? createObjectPropertyFromRequestBody(requestBody) ?? []
                    : [],
            ),
    };

    const endpointContext = {
        path,
        method,
        original,
    };

    return {
        path,
        method,
        original,
        display: {
            parametersName: configuration.resolve.endpointParametersName(endpointContext),
            responseName: configuration.resolve.endpointSuccessResponseName(endpointContext),
            methodName: configuration.resolve.endpointMethodName(endpointContext),
            fullName: configuration.resolve.endpointFullName(endpointContext),
        },
        nodeType: 'endpoint',
        parameters,
        responses: createEndpointResponsesNode(original.responses ?? {}, configuration),
        requestBody,
        virtualParametersSchema,
    };
}

export function createEndpointResponsesNode(
    original: OpenAPIV3.ResponsesObject,
    configuration: Configuration,
): EndpointResponsesNode {
    const all = Object.entries(original).map(([name, value]) =>
        createEndpointResponseNode(name, createResponseNode(value, configuration), configuration),
    );

    return {
        nodeType: 'endpoint-responses',
        original,
        success: all.filter(node => node.success),
        failed: all.filter(node => !node.success),
        all,
    };
}

export function createEndpointResponseNode(
    name: string,
    node: AnyResponseNode,
    configuration: Configuration,
): EndpointResponseNode {
    return {
        nodeType: 'endpoint-response',
        name,
        value: node,
        success: configuration.resolve.isSuccessResponseCode(name),
    };
}

const createObjectPropertyFromRequestBody = ({
    media,
    original: { required },
}: RequestBodyNode): ObjectPropertyNode | null =>
    media.primary
        ? {
              nodeType: 'property',
              value: media.primary.schema,
              // TODO Добавить в конфигурацию
              name: 'body',
              nullable: false,
              optional: !required,
          }
        : null;

const createObjectPropertyFromParameter = ({
    schema,
    name,
    original: { required },
}: ParameterNode): ObjectPropertyNode => ({
    nodeType: 'property',
    value: schema,
    name,
    nullable: false,
    optional: !required,
});

const SUPPORTED_HTTP_METHODS: OpenAPIV3.HttpMethods[] = [
    OpenAPIV3.HttpMethods.GET,
    OpenAPIV3.HttpMethods.PUT,
    OpenAPIV3.HttpMethods.POST,
    OpenAPIV3.HttpMethods.PATCH,
    OpenAPIV3.HttpMethods.DELETE,
    // OpenAPIV3.HttpMethods.HEAD,
    // OpenAPIV3.HttpMethods.TRACE,
    // OpenAPIV3.HttpMethods.OPTIONS,
];
