import { OpenAPIV3 } from 'openapi-types';
import { Configuration } from '../configuration';
import {
    AnyParameterNode,
    AnyRequestBodyNode,
    AnyResponseNode,
    AnySchemaNode,
    ComponentType,
    ComponentTypeToNode,
    ComponentTypeToObject,
    DocumentNodeResolved,
    EndpointNode,
    EndpointResponseNode,
    ReferenceNode,
} from './types';

type AnyNode =
    | AnyParameterNode
    | AnyResponseNode
    | AnyRequestBodyNode
    | AnySchemaNode
    | EndpointNode
    | EndpointResponseNode;

// TODO Разделить на этапы 1) получаем компоненты без ссылок 2) резолвим ссылки в компонентах 3) резолвим остальные ссылки

export function traverseNodeReferences(
    node: AnyNode,
    resolved: DocumentNodeResolved,
    ignoreRecursiveReferences = true,
    collected: ReferenceNode[] = [],
) {
    const recurse = (next: AnyNode) =>
        traverseNodeReferences(next, resolved, ignoreRecursiveReferences, collected);

    switch (node.nodeType) {
        case 'ref':
            if (!node.resolved) {
                node.resolved = resolved[node.type][node.name];
            }

            if (!collected.includes(node)) {
                collected.push(node);

                if (!ignoreRecursiveReferences) {
                    traverseNodeReferences(node.resolved, resolved, false, collected);
                }
            }

            break;
        case 'endpoint':
            node.parameters.map(recurse);
            node.responses.all.map(recurse);

            if (node.requestBody) {
                recurse(node.requestBody);
            }

            break;
        case 'endpoint-response':
            recurse(node.value);

            break;
        case 'parameter':
            recurse(node.schema);

            break;
        case 'response':
        case 'request-body':
            node.media.all.map(media => recurse(media.schema));

            break;
        case 'schema':
            node.combined?.map(({ value }) => value.map(recurse));

            if (node.schemaType === 'array') {
                recurse(node.value);
            }

            if (node.schemaType === 'object') {
                node.properties.map(prop => recurse(prop.value));

                if (node.index && node.index !== 'unknown') {
                    recurse(node.index);
                }
            }

            break;

        default:
            break;
    }

    return collected;
}

/**
 * TODO Добавить резолвинг всех ссылок
 * Resolves all references. Required for merge specific rules (ex. nullable)
 * { foo: { $ref: bar }, bar: { $ref: baz }, baz: { ... } }
 * becomes
 * { foo: { ..., resolved: baz }, bar: { ..., resolved: baz }, ... }
 */
export function resolveTopLevelReferences<Type extends ComponentType>(
    target: Record<string, ComponentTypeToNode[Type] | ReferenceNode<Type>>,
): Record<string, ComponentTypeToNode[Type]> {
    return Object.fromEntries(
        Object.entries(target).map(([name, value]) => [
            name,
            getOrResolveTopLevelReference(value, target),
        ]),
    );
}

export const getResolved = <Type extends ComponentType>(
    value: ComponentTypeToNode[Type] | ReferenceNode<Type>,
) => (value.nodeType === 'ref' ? value.resolved! : value);

export const getOrResolveReference = <Type extends ComponentType>(
    value: ComponentTypeToNode[Type] | ReferenceNode<Type>,
    resolved: Record<string, ComponentTypeToNode[Type]>,
) => (value.nodeType === 'ref' ? resolved[value.name] : value);

export function getOrResolveTopLevelReference<
    Type extends ComponentType,
    Ref extends ReferenceNode<Type>,
>(
    value: ComponentTypeToNode[Type] | Ref,
    registry: Record<string, ComponentTypeToNode[Type] | Ref>,
): ComponentTypeToNode[Type] {
    if (isReferenceNode(value)) {
        if (!value.resolved) {
            value.resolved = getOrResolveTopLevelReference(registry[value.name], registry);
        }

        return value.resolved;
    }

    return value;
}

// eslint-disable-next-line @typescript-eslint/ban-types
export function createReferenceFactory<Type extends ComponentType, FactoryArgs extends any[] = []>(
    fn: (
        value: Exclude<ComponentTypeToObject[Type], OpenAPIV3.ReferenceObject>,
        configuration: Configuration,
        ...args: FactoryArgs
    ) => ComponentTypeToNode[Type],
) {
    return function factory(
        value: ComponentTypeToObject[Type] | OpenAPIV3.ReferenceObject,
        configuration: Configuration,
        ...args: FactoryArgs
    ): ComponentTypeToNode[Type] | ReferenceNode<Type> {
        return isReference(value)
            ? createReferenceNode<Type>(value, configuration)
            : // TODO fixme
              fn(value as any, configuration, ...args);
    };
}

export function createReferenceNode<Type extends ComponentType>(
    original: OpenAPIV3.ReferenceObject,
    configuration: Configuration,
): ReferenceNode<Type> {
    const { type, name } = referenceRe.exec(original.$ref)?.groups ?? {};

    return {
        nodeType: 'ref',
        original,
        type: type as Type,
        name: resolveReferenceName(name, type as Type, configuration),
    };
}

// eslint-disable-next-line @typescript-eslint/ban-types
export const isReference = <T extends {}>(
    value: T | OpenAPIV3.ReferenceObject,
): value is OpenAPIV3.ReferenceObject => Object.hasOwn(value, '$ref');

export const isReferenceNode = <Type extends ComponentType>(
    target: ComponentTypeToNode[Type] | ReferenceNode<Type>,
): target is ReferenceNode<Type> => target.nodeType === 'ref';

export const resolveReferenceName = (
    name: string,
    type: ComponentType,
    { resolve }: Configuration,
) => {
    switch (type) {
        case 'parameters':
            return resolve.parameterName(name);
        case 'responses':
            return resolve.responseName(name);
        case 'requestBodies':
            return resolve.requestBodyName(name);
        default:
            return resolve.schemaName(name);
    }
};

const referenceRe = /^#\/components\/(?<type>[a-z]*)\/(?<name>.*)$/;
