import { OpenAPIV3 } from 'openapi-types';
import type { OutputConfiguration, OutputPathInfo, SourcePathInfo } from '../configuration';

/**
 * Aggregation
 */

export interface ComponentTypeToNode {
    schemas: SchemaNode;
    responses: ResponseNode;
    parameters: ParameterNode;
    requestBodies: RequestBodyNode;
}

/**
 * Document
 */

export interface DocumentNode extends TreeNode<'document'> {
    info: DocumentInfo;
    original: OpenAPIV3.Document;
    resolved: DocumentNodeResolved;
    endpoints: EndpointsPathNode[];
    components: DocumentNodeComponents;
}

// TODO Добавить возможность динамически группировать эндпоинты по сервисам, а не по целому документу
export interface DocumentInfo {
    /**
     * Полное имя документа, используется для генерации прикладного слоя
     * @example AdministrationAPI
     */
    fullName: string;
    /**
     * Информация о пути до сгенерированных файлов
     * @example { relativeToOutput: "./my-api", relativeToCwd: "./generated/my-api", absolute: "..." }
     */
    output: Record<keyof OutputConfiguration, OutputPathInfo>;
    source: SourcePathInfo;
}

export interface DocumentNodeComponents {
    schemas: Record<string, AnySchemaNode>;
    responses: Record<string, AnyResponseNode>;
    parameters: Record<string, AnyParameterNode>;
    requestBodies: Record<string, AnyRequestBodyNode>;
}

export interface DocumentNodeResolved {
    schemas: Record<string, SchemaNode>;
    responses: Record<string, ResponseNode>;
    parameters: Record<string, ParameterNode>;
    requestBodies: Record<string, RequestBodyNode>;
}

/**
 * Endpoints
 */

export interface EndpointsPathNode extends TreeNode<'endpoints-path'> {
    path: string;
    original: OpenAPIV3.PathItemObject;
    endpoints: EndpointNode[];
}

export interface EndpointNode extends TreeNode<'endpoint'> {
    original: OpenAPIV3.OperationObject;
    display: EndpointDisplayName;
    path: string;
    method: OpenAPIV3.HttpMethods;
    responses: EndpointResponsesNode;
    parameters: AnyParameterNode[];
    requestBody: AnyRequestBodyNode | null;
    /**
     * Динамически создаем объект со всеми параметрами
     */
    virtualParametersSchema: ObjectSchemaNode;
}

export interface EndpointDisplayName {
    parametersName: string;
    responseName: string;
    /**
     * Короткое имя метода, должно использоваться для генерации прикладного кода
     * @example createNewResolution
     */
    methodName: string;
    /**
     * Полный идентификатор операции, должен использоваться для генерации имён
     * @example FooBarService/get-my-dear-friend-description
     */
    fullName: string;
}

export interface EndpointResponsesNode extends TreeNode<'endpoint-responses'> {
    original: OpenAPIV3.ResponsesObject;
    success: EndpointResponseNode[];
    failed: EndpointResponseNode[];
    all: EndpointResponseNode[];
}

export interface EndpointResponseNode extends TreeNode<'endpoint-response'> {
    success: boolean;
    value: AnyResponseNode;
    name: string;
}

/**
 * Responses
 */

export type AnyResponseNode = ResponseNode | ReferenceResponseNode;
export type ReferenceResponseNode = ReferenceNode<'responses'>;

export interface ResponseNode extends TreeNode<'response'> {
    virtualSchemaNode: AnySchemaNode;
    original: OpenAPIV3.ResponseObject;
    media: MediaVariantsNode;
}

/**
 * Parameters
 */

export type AnyParameterNode = ParameterNode | ReferenceParameterNode;
export type ReferenceParameterNode = ReferenceNode<'parameters'>;

export interface ParameterNode extends TreeNode<'parameter'> {
    virtualSchemaNode: AnySchemaNode;
    original: OpenAPIV3.ParameterObject;
    schema: AnySchemaNode;
    name: string;
    type: ParameterType;
}

/**
 * Request bodies
 */

export type AnyRequestBodyNode = RequestBodyNode | ReferenceRequestBodyNode;
export type ReferenceRequestBodyNode = ReferenceNode<'requestBodies'>;

export interface RequestBodyNode extends TreeNode<'request-body'> {
    virtualSchemaNode: AnySchemaNode;
    original: OpenAPIV3.RequestBodyObject;
    media: MediaVariantsNode;
}

/**
 * Media type
 */

export interface MediaVariantsNode extends TreeNode<'media-variants'> {
    original: Record<string, OpenAPIV3.MediaTypeObject>;
    primary: MediaTypeNode | null;
    all: MediaTypeNode[];
}

export interface MediaTypeNode extends TreeNode<'media-type'> {
    original: OpenAPIV3.MediaTypeObject;
    schema: AnySchemaNode;
    name: string;
}

/**
 * Schemas
 */

export type AnySchemaNode = SchemaNode | ReferenceSchemaNode;
export type SchemaNode =
    | EnumSchemaNode
    | ArraySchemaNode
    | ObjectSchemaNode
    | PrimitiveSchemaNode
    | UnknownSchemaNode;
export type ReferenceSchemaNode = ReferenceNode<'schemas'>;

// == unknown ==

export type UnknownSchemaNode = BaseSchemaNode<'unknown', OpenAPIV3.NonArraySchemaObject>;

// == primitive ==

export type PrimitiveSchemaNode = BaseSchemaNode<
    PrimitiveSchemaType,
    OpenAPIV3.NonArraySchemaObject
>;

// == object ==

export interface ObjectSchemaNode extends BaseSchemaNode<'object', OpenAPIV3.NonArraySchemaObject> {
    properties: ObjectPropertyNode[];
    index: ObjectSchemaIndexType | null;
}

export type ObjectSchemaIndexType = AnySchemaNode | 'unknown';

export interface ObjectPropertyNode extends TreeNode<'property'> {
    name: string;
    value: AnySchemaNode;
    /**
     * foo: number | null;
     */
    nullable: boolean;
    /**
     * foo?: number;
     */
    optional: boolean;
}

// == array ==

export interface ArraySchemaNode extends BaseSchemaNode<'array', OpenAPIV3.ArraySchemaObject> {
    value: AnySchemaNode;
}

// == enum ==

export interface EnumSchemaNode extends BaseSchemaNode<'enum', OpenapiEnumObject> {
    properties: EnumSchemaProperty[];
}

export interface EnumSchemaProperty {
    name: string;
    value: string | number;
}

// == base ==

export interface BaseSchemaNode<Type extends SchemaType, Original extends OpenAPIV3.SchemaObject>
    extends TreeNode<'schema'> {
    schemaType: Type;
    original: Original;
    combined: SchemaCombinedInfo[] | null;
}

export interface SchemaCombinedInfo {
    type: SchemaCombinedType;
    value: AnySchemaNode[];
}

/**
 * Core and shared
 */

export interface ReferenceNode<Type extends ComponentType = ComponentType> extends TreeNode<'ref'> {
    type: Type;
    name: string;
    resolved?: ComponentTypeToNode[Type];
    original: OpenAPIV3.ReferenceObject;
}

export interface TreeNode<Type> {
    nodeType: Type;
}

export type ParameterType = 'query' | 'path' | 'header' | 'cookie';

export type SchemaCombinedType = 'allOf' | 'anyOf' | 'oneOf';

export type SchemaType = PrimitiveSchemaType | 'object' | 'enum' | 'array' | 'unknown';

export type PrimitiveSchemaType = 'string' | 'number' | 'integer' | 'boolean';

export type ComponentType = Exclude<
    keyof Required<OpenAPIV3.ComponentsObject>,
    'securitySchemes' | 'headers' | 'callbacks' | 'links' | 'examples'
>;

export type ComponentTypeToObject = {
    [Type in keyof Required<OpenAPIV3.ComponentsObject>]: Required<OpenAPIV3.ComponentsObject>[Type] extends {
        [key: string]: infer Value;
    }
        ? Value
        : never;
};

/**
 * Openapi extensions
 */

export interface OpenapiEnumObject extends OpenAPIV3.NonArraySchemaObject {
    enum: Array<string | number>;
    'x-enumNames'?: string[];
}
